/*
* Copyright (c) 2013-2026, The PurpleI2P Project
*
* This file is part of Purple i2pd project and licensed under BSD3
*
* See full license text in LICENSE file at top of project tree
*/

#ifndef SAM_H__
#define SAM_H__

#include <inttypes.h>
#include <string>
#include <string_view>
#include <map>
#include <list>
#include <set>
#include <thread>
#include <mutex>
#include <memory>
#include <boost/asio.hpp>
#include "util.h"
#include "Identity.h"
#include "LeaseSet.h"
#include "Streaming.h"
#include "Datagram.h"
#include "Destination.h"

namespace i2p
{
namespace client
{
	const size_t SAM_SOCKET_BUFFER_SIZE = 8192;
	const size_t SAM_STREAM_BUFFER_SIZE = 16384;
	const size_t SAM_STREAM_MAX_SEND_BUFFER_SIZE = 8*SAM_SOCKET_BUFFER_SIZE;
	const int SAM_SOCKET_CONNECTION_MAX_IDLE = 3600; // in seconds
	const int SAM_SESSION_READINESS_CHECK_INTERVAL = 3; // in seconds
	const size_t SAM_SESSION_MAX_ACCEPT_QUEUE_SIZE = 50;
	const size_t SAM_SESSION_MAX_ACCEPT_INTERVAL = 3; // in seconds

	const char SAM_HANDSHAKE[] = "HELLO VERSION";
	const char SAM_HANDSHAKE_REPLY[] = "HELLO REPLY RESULT=OK VERSION=%s\n";
	constexpr std::string_view SAM_HANDSHAKE_NOVERSION { "HELLO REPLY RESULT=NOVERSION\n" };
	const char SAM_HANDSHAKE_I2P_ERROR[] = "HELLO REPLY RESULT=I2P_ERROR\n";
	const char SAM_SESSION_CREATE[] = "SESSION CREATE";
	const char SAM_SESSION_CREATE_REPLY_OK[] = "SESSION STATUS RESULT=OK DESTINATION=%s\n";
	constexpr std::string_view SAM_SESSION_CREATE_DUPLICATED_ID { "SESSION STATUS RESULT=DUPLICATED_ID\n" };
	constexpr std::string_view SAM_SESSION_CREATE_DUPLICATED_DEST { "SESSION STATUS RESULT=DUPLICATED_DEST\n" };
	constexpr std::string_view SAM_SESSION_CREATE_INVALID_ID { "SESSION STATUS RESULT=INVALID_ID\n" };
	constexpr std::string_view SAM_SESSION_STATUS_INVALID_KEY { "SESSION STATUS RESULT=INVALID_KEY\n" };
	const char SAM_SESSION_STATUS_I2P_ERROR[] = "SESSION STATUS RESULT=I2P_ERROR MESSAGE=\"%s\"\n";
	const char SAM_SESSION_ADD[] = "SESSION ADD";
	const char SAM_SESSION_REMOVE[] = "SESSION REMOVE";
	const char SAM_STREAM_CONNECT[] = "STREAM CONNECT";
	constexpr std::string_view SAM_STREAM_STATUS_OK { "STREAM STATUS RESULT=OK\n" };
	constexpr std::string_view SAM_STREAM_STATUS_INVALID_ID { "STREAM STATUS RESULT=INVALID_ID\n" };
	constexpr std::string_view SAM_STREAM_STATUS_INVALID_KEY { "STREAM STATUS RESULT=INVALID_KEY\n" };
	const char SAM_STREAM_STATUS_CANT_REACH_PEER[] = "STREAM STATUS RESULT=CANT_REACH_PEER MESSAGE=\"%s\"\n";
	const char SAM_STREAM_STATUS_I2P_ERROR[] = "STREAM STATUS RESULT=I2P_ERROR MESSAGE=\"%s\"\n";
	const char SAM_STREAM_ACCEPT[] = "STREAM ACCEPT";
	const char SAM_STREAM_FORWARD[] = "STREAM FORWARD";
	const char SAM_DATAGRAM_SEND[] = "DATAGRAM SEND";
	const char SAM_RAW_SEND[] = "RAW SEND";
	const char SAM_DEST_GENERATE[] = "DEST GENERATE";
	const char SAM_DEST_REPLY[] = "DEST REPLY PUB=%s PRIV=%s\n";
	const char SAM_DEST_REPLY_I2P_ERROR[] = "DEST REPLY RESULT=I2P_ERROR\n";
	const char SAM_NAMING_LOOKUP[] = "NAMING LOOKUP";
	const char SAM_NAMING_REPLY[] = "NAMING REPLY RESULT=OK NAME=%s VALUE=%s\n";
	const char SAM_DATAGRAM_RECEIVED[] = "DATAGRAM RECEIVED DESTINATION=%s SIZE=%lu FROM_PORT=%u TO_PORT=%u\n";
	const char SAM_RAW_RECEIVED[] = "RAW RECEIVED SIZE=%lu\n";
	const char SAM_NAMING_REPLY_INVALID_KEY[] = "NAMING REPLY RESULT=INVALID_KEY NAME=%s\n";
	const char SAM_NAMING_REPLY_KEY_NOT_FOUND[] = "NAMING REPLY RESULT=KEY_NOT_FOUND NAME=%s\n";
	const char SAM_PARAM_MIN[] = "MIN";
	const char SAM_PARAM_MAX[] = "MAX";
	const char SAM_PARAM_STYLE[] = "STYLE";
	const char SAM_PARAM_ID[] = "ID";
	const char SAM_PARAM_SILENT[] = "SILENT";
	const char SAM_PARAM_DESTINATION[] = "DESTINATION";
	const char SAM_PARAM_NAME[] = "NAME";
	const char SAM_PARAM_SIGNATURE_TYPE[] = "SIGNATURE_TYPE";
	const char SAM_PARAM_CRYPTO_TYPE[] = "CRYPTO_TYPE";
	const char SAM_PARAM_SIZE[] = "SIZE";
	const char SAM_PARAM_HOST[] = "HOST";
	const char SAM_PARAM_PORT[] = "PORT";
	const char SAM_PARAM_FROM_PORT[] = "FROM_PORT";
	const char SAM_PARAM_TO_PORT[] = "TO_PORT";
	const char SAM_VALUE_TRANSIENT[] = "TRANSIENT";
	const char SAM_VALUE_TRUE[] = "true";
	const char SAM_VALUE_FALSE[] = "false";

	constexpr std::string_view SAM_VALUE_STREAM { "STREAM" };
	constexpr std::string_view SAM_VALUE_DATAGRAM { "DATAGRAM" };
	constexpr std::string_view SAM_VALUE_RAW { "RAW" };
	constexpr std::string_view SAM_VALUE_MASTER { "MASTER" };

	constexpr std::string_view SAM_PING { "PING" };
	const char SAM_PONG[] = "PONG %s\n";

	constexpr int MAKE_SAM_VERSION_NUMBER (int major, int minor) { return major*10 + minor; }
	constexpr int MIN_SAM_VERSION = MAKE_SAM_VERSION_NUMBER (3, 0);
	constexpr int MAX_SAM_VERSION = MAKE_SAM_VERSION_NUMBER (3, 3);
	constexpr int SAM_VERSION_33 = MAKE_SAM_VERSION_NUMBER (3, 3); // SAM 3.3

	enum class SAMSocketType
	{
		eSAMSocketTypeUnknown,
		eSAMSocketTypeSession,
		eSAMSocketTypeStream,
		eSAMSocketTypeAcceptor,
		eSAMSocketTypeForward,
		eSAMSocketTypeTerminated
	};

	class SAMBridge;
	struct SAMSession;
	class SAMSocket: public std::enable_shared_from_this<SAMSocket>
	{
		public:

			typedef boost::asio::ip::tcp::socket Socket_t;
			SAMSocket (SAMBridge& owner);
			~SAMSocket ();

			Socket_t& GetSocket () { return m_Socket; };
			void ReceiveHandshake ();
			void SetSocketType (SAMSocketType socketType) { m_SocketType = socketType; };
			SAMSocketType GetSocketType () const { return m_SocketType; };

			void Terminate (const char* reason);

			bool IsSession(std::string_view id) const;

			static i2p::util::Mapping ExtractParams (std::string_view buf);

		private:

			void TerminateClose() { Terminate(nullptr); }

			void HandleHandshakeReceived (const boost::system::error_code& ecode, std::size_t bytes_transferred);
			void HandleHandshakeReplySent (const boost::system::error_code& ecode, std::size_t bytes_transferred);
			void HandleMessage (const boost::system::error_code& ecode, std::size_t bytes_transferred);
			void SendMessageReply (std::string_view msg, bool close);
			void HandleMessageReplySent (const boost::system::error_code& ecode, std::size_t bytes_transferred, bool close);
			void Receive ();
			void HandleReceived (const boost::system::error_code& ecode, std::size_t bytes_transferred);

			void I2PReceive ();
			void HandleI2PReceive (const boost::system::error_code& ecode, std::size_t bytes_transferred);
			void HandleI2PAccept (std::shared_ptr<i2p::stream::Stream> stream);
			void HandleI2PForward (std::shared_ptr<i2p::stream::Stream> stream, boost::asio::ip::tcp::endpoint ep);
			void HandleWriteI2PData (const boost::system::error_code& ecode, size_t sz);
			void HandleI2PDatagramReceive (const i2p::data::IdentityEx& from, uint16_t fromPort, uint16_t toPort,
				const uint8_t * buf, size_t len, const i2p::util::Mapping * options);
			void HandleI2PRawDatagramReceive (uint16_t fromPort, uint16_t toPort, const uint8_t * buf, size_t len);

			void ProcessSessionCreate (std::string_view buf);
			void ProcessStreamConnect (char * buf, size_t len, size_t rem);
			void ProcessStreamAccept (std::string_view buf);
			void ProcessStreamForward (std::string_view buf);
			void ProcessDestGenerate (std::string_view buf);
			void ProcessNamingLookup (std::string_view buf);
			void ProcessSessionAdd (std::string_view buf);
			void ProcessSessionRemove (std::string_view buf);
			void ProcessPing (std::string_view text);
			void SendReplyWithMessage (const char * reply, const std::string & msg);
			void SendSessionI2PError(const std::string & msg);
			void SendStreamI2PError(const std::string & msg);
			void SendStreamCantReachPeer(const std::string & msg);
			size_t ProcessDatagramSend (char * buf, size_t len, const char * data); // from SAM 1.0

			void Connect (std::shared_ptr<const i2p::data::LeaseSet> remote, std::shared_ptr<SAMSession> session = nullptr);
			void HandleConnectLeaseSetRequestComplete (std::shared_ptr<i2p::data::LeaseSet> leaseSet);
			void SendNamingLookupReply (const std::string& name, std::shared_ptr<const i2p::data::IdentityEx> identity);
			void HandleNamingLookupLeaseSetRequestComplete (std::shared_ptr<i2p::data::LeaseSet> leaseSet, std::string name);
			void HandleSessionReadinessCheckTimer (const boost::system::error_code& ecode);
			void SendSessionCreateReplyOk ();

			void WriteI2PData(size_t sz);
			void WriteI2PDataImmediate(uint8_t * ptr, size_t sz);

			void HandleWriteI2PDataImmediate(const boost::system::error_code & ec, uint8_t * buff);
			void HandleStreamSend(const boost::system::error_code & ec);

		private:

			SAMBridge& m_Owner;
			Socket_t m_Socket;
			boost::asio::deadline_timer m_Timer;
			char m_Buffer[SAM_SOCKET_BUFFER_SIZE + 1];
			size_t m_BufferOffset; // for session only
			uint8_t m_StreamBuffer[SAM_STREAM_BUFFER_SIZE];
			SAMSocketType m_SocketType;
			std::string m_ID; // nickname
			bool m_IsSilent;
			bool m_IsAccepting; // for eSAMSocketTypeAcceptor only
			bool m_IsReceiving; // for eSAMSocketTypeStream only
			std::shared_ptr<i2p::stream::Stream> m_Stream;
			int m_Version;
	};

	enum class SAMSessionType
	{
		eSAMSessionTypeUnknown,
		eSAMSessionTypeStream,
		eSAMSessionTypeDatagram,
		eSAMSessionTypeRaw,
		eSAMSessionTypeMaster
	};

	struct SAMSession
	{
		SAMBridge & m_Bridge;
		std::string Name;
		SAMSessionType Type;
		i2p::datagram::DatagramVersion DatagramVersion;

		std::shared_ptr<boost::asio::ip::udp::endpoint> UDPEndpoint; // TODO: move
		std::list<std::pair<std::shared_ptr<SAMSocket>, uint64_t> > acceptQueue; // socket, receive time in seconds

		SAMSession (SAMBridge & parent, std::string_view name, SAMSessionType type);
		virtual ~SAMSession () {};

		virtual std::shared_ptr<ClientDestination> GetLocalDestination () = 0;
		virtual void StopLocalDestination () = 0;
		virtual void Close () { CloseStreams (); };

		void CloseStreams ();
	};

	struct SAMSingleSession: public SAMSession
	{
		std::shared_ptr<ClientDestination> localDestination;

		SAMSingleSession (SAMBridge & parent, std::string_view name, SAMSessionType type, std::shared_ptr<ClientDestination> dest);
		~SAMSingleSession ();

		std::shared_ptr<ClientDestination> GetLocalDestination () { return localDestination; };
		void StopLocalDestination ();
	};

	struct SAMMasterSession: public SAMSingleSession
	{
		std::set<std::string, std::less<> > subsessions;
		SAMMasterSession (SAMBridge & parent, std::string_view name, std::shared_ptr<ClientDestination> dest):
			SAMSingleSession (parent, name, SAMSessionType::eSAMSessionTypeMaster, dest) {};
		void Close ();
	};

	struct SAMSubSession: public SAMSession
	{
		std::shared_ptr<SAMMasterSession> masterSession;
		uint16_t inPort;

		SAMSubSession (std::shared_ptr<SAMMasterSession> master, std::string_view name, SAMSessionType type, uint16_t port);
		// implements SAMSession
		std::shared_ptr<ClientDestination> GetLocalDestination ();
		void StopLocalDestination ();
	};

	class SAMBridge: private i2p::util::RunnableService
	{
		public:

			SAMBridge (const std::string& address, uint16_t portTCP, uint16_t portUDP, bool singleThread);
			~SAMBridge ();

			void Start ();
			void Stop ();

			auto& GetService () { return GetIOService (); };
			std::shared_ptr<SAMSession> CreateSession (std::string_view id, SAMSessionType type, std::string_view destination, // empty string means transient
				const i2p::util::Mapping& params);
			bool AddSession (std::shared_ptr<SAMSession> session);
			void CloseSession (std::string_view id);
			std::shared_ptr<SAMSession> FindSession (std::string_view id) const;

			std::list<std::shared_ptr<SAMSocket> > ListSockets(std::string_view id) const;

			/** send raw data to remote endpoint from our UDP Socket */
			void SendTo (const std::vector<boost::asio::const_buffer>& bufs, const boost::asio::ip::udp::endpoint& ep);

			void AddSocket(std::shared_ptr<SAMSocket> socket);
			void RemoveSocket(const std::shared_ptr<SAMSocket> & socket);

			bool ResolveSignatureType (std::string_view name, i2p::data::SigningKeyType& type) const;

		private:

			void Accept ();
			void HandleAccept(const boost::system::error_code& ecode, std::shared_ptr<SAMSocket> socket);

			void ReceiveDatagram ();
			void HandleReceivedDatagram (const boost::system::error_code& ecode, std::size_t bytes_transferred);

			void ScheduleSessionCleanupTimer (std::shared_ptr<SAMSession> session);
			void HandleSessionCleanupTimer (const boost::system::error_code& ecode,
				std::shared_ptr<SAMSession> session, std::shared_ptr<boost::asio::deadline_timer> timer);

		private:

			bool m_IsSingleThread;
			boost::asio::ip::tcp::acceptor m_Acceptor;
			boost::asio::ip::udp::endpoint m_DatagramEndpoint, m_SenderEndpoint;
			boost::asio::ip::udp::socket m_DatagramSocket;
			mutable std::mutex m_SessionsMutex;
			std::map<std::string, std::shared_ptr<SAMSession>, std::less<>> m_Sessions;
			mutable std::mutex m_OpenSocketsMutex;
			std::list<std::shared_ptr<SAMSocket> > m_OpenSockets;
			uint8_t m_DatagramReceiveBuffer[i2p::datagram::MAX_DATAGRAM_SIZE+1];
			const std::map<std::string_view, i2p::data::SigningKeyType> m_SignatureTypes;

		public:

			// for HTTP
			const decltype(m_Sessions)& GetSessions () const { return m_Sessions; };
	};
}
}

#endif
