package Color::Library::Dictionary::NBS_ISCC::R;

use strict;
use warnings;

use base qw/Color::Library::Dictionary/;

__PACKAGE__->_register_dictionary;

package Color::Library::Dictionary::NBS_ISCC::R;

=pod

=head1 NAME

Color::Library::Dictionary::NBS_ISCC::R - (NBS/ISCC R) Color Standards and Color Nomenclature

=head1 DESCRIPTION

    Kenneth L. Kelly and Deanne B. Judd.
    "Color: Universal Language and Dictionary of Names",
    National Bureau of Standards,
    Spec. Publ. 440, Dec. 1976, 189 pages.


Color Name Dictionary: (R) Color Standards and Color Nomenclature

Ridgway,
Color Standards and Color Nomenclature 

While using some modifiers, this primarily idiosyncratic dictionary was intended for biology and botany. It maps 1096 names to 1602 colors, using 214 of the 267 centroids.

L<http://swissnet.ai.mit.edu/~jaffer/Color/Dictionaries#R>

=head1 COLORS

	absinthe green             absinthegreen           #8a9a5b

	acajou red                 acajoured               #905d5d

	acetin blue                acetinblue              #367588

	acetin blue                acetinblue              #4c516d

	ackermann's green          ackermannsgreen         #3b7861

	aconite violet             aconiteviolet           #86608e

	ageratum violet            ageratumviolet          #86608e

	alice blue                 aliceblue               #91a3b0

	alizarine blue             alizarineblue           #436b95

	alizarine pink             alizarinepink           #ea9399

	alizarine pink             alizarinepink           #e4717a

	amaranth pink              amaranthpink            #de6fa1

	amaranth purple            amaranthpurple          #78184a

	amber brown                amberbrown              #80461b

	amber yellow               amberyellow             #fada5e

	amber yellow               amberyellow             #d4af37

	amber yellow               amberyellow             #f8de7e

	amber yellow               amberyellow             #c9ae5d

	american green             americangreen           #679267

	amethyst violet            amethystviolet          #9a4eae

	amparo blue                amparoblue              #6c79b8

	amparo purple              amparopurple            #d399e6

	amparo purple              amparopurple            #875692

	andover green              andovergreen            #8f9779

	aniline black              anilineblack            #50404d

	aniline black              anilineblack            #5d555b

	aniline lilac              anilinelilac            #8c82b6

	aniline yellow             anilineyellow           #af8d13

	anthracene green           anthracenegreen         #317873

	anthracene green           anthracenegreen         #004b49

	anthracene purple          anthracenepurple        #50404d

	anthracene purple          anthracenepurple        #673147

	anthracene violet          anthraceneviolet        #604e81

	antimony yellow            antimonyyellow          #e3a857

	antique brown              antiquebrown            #80461b

	antique brown              antiquebrown            #6f4e37

	antique green              antiquegreen            #5e716a

	antique green              antiquegreen            #3a4b47

	antwerp blue               antwerpblue             #436b95

	antwerp blue               antwerpblue             #00304e

	antwerp blue               antwerpblue             #536878

	apple green                applegreen              #7e9f2e

	apricot buff               apricotbuff             #d99058

	apricot orange             apricotorange           #d99058

	apricot yellow             apricotyellow           #d4af37

	argus brown                argusbrown              #6f4e37

	argyle purple              argylepurple            #915c83

	army brown                 armybrown               #6f4e37

	army brown                 armybrown               #635147

	artemesia green            artemesiagreen          #7d8984

	asphodel green             asphodelgreen           #8a9a5b

	aster purple               asterpurple             #78184a

	auburn                     auburn                  #79443b

	auricula purple            auriculapurple          #673147

	avellaneous                avellaneous             #c19a6b

	avellaneous                avellaneous             #ae9b82

	azurite blue               azuriteblue             #4e5180

	barium yellow              bariumyellow            #eae679

	barium yellow              bariumyellow            #b9b459

	baryta yellow              barytayellow            #f8de7e

	bay                        bay                     #79443b

	begonia rose               begoniarose             #e66761

	benzo brown                benzobrown              #977f73

	benzo brown                benzobrown              #958070

	benzo brown                benzobrown              #635147

	benzol green               benzolgreen             #007a74

	berlin blue                berlinblue              #00304e

	beryl blue                 berylblue               #9cd1dc

	beryl green                berylgreen              #00a693

	bice green                 bicegreen               #8a9a5b

	biscay green               biscaygreen             #8a9a5b

	bishop's purple            bishopspurple           #915c83

	bister                     bister                  #7e6d5a

	bister                     bister                  #483c32

	bittersweet orange         bittersweetorange       #cb6d51

	bittersweet pink           bittersweetpink         #d99058

	black                      black                   #222222

	blackish green-blue        blackishgreenblue       #004958

	blackish green-gray        blackishgreengray       #222222

	blackish mouse-gray        blackishmousegray       #5b504f

	blackish mouse-gray        blackishmousegray       #555555

	blackish plumbeous         blackishplumbeous       #51585e

	blackish purple            blackishpurple          #5d3954

	blackish purple            blackishpurple          #341731

	blackish red-purple        blackishredpurple       #673147

	blackish slate             blackishslate           #555555

	blackish violet            blackishviolet          #604e81

	blackish violet            blackishviolet          #2f2140

	blackish violet            blackishviolet          #554c69

	blackish violet-gray       blackishvioletgray      #5d555b

	blackish violet-gray       blackishvioletgray      #555555

	blanc's blue               blancsblue              #436b95

	blanc's violet             blancsviolet            #554c69

	blanc's violet             blancsviolet            #796878

	blanc's violet             blancsviolet            #50404d

	blue-violet                blueviolet              #30267a

	blue-violet                blueviolet              #9065ca

	blue-violet black          bluevioletblack         #5d555b

	blue-violet black          bluevioletblack         #555555

	bluish black               bluishblack             #36454f

	bluish black               bluishblack             #202830

	bluish glaucous            bluishglaucous          #c7e6d7

	bluish glaucous            bluishglaucous          #96ded1

	bluish gray-green          bluishgraygreen         #8da399

	bluish gray-green          bluishgraygreen         #5e716a

	bluish lavender            bluishlavender          #8c82b6

	bluish slate-black         bluishslateblack        #00304e

	bluish slate-black         bluishslateblack        #36454f

	bluish violet              bluishviolet            #30267a

	bluish violet              bluishviolet            #9065ca

	bone brown                 bonebrown               #635147

	bone brown                 bonebrown               #5b504f

	bordeaux                   bordeaux                #673147

	bottle green               bottlegreen             #1b4d3e

	bradley's blue             bradleysblue            #00a1c2

	bradley's violet           bradleysviolet          #7e73b8

	bradley's violet           bradleysviolet          #604e97

	bradley's violet           bradleysviolet          #8c82b6

	bradley's violet           bradleysviolet          #604e81

	brazil red                 brazilred               #9e4732

	bremen blue                bremenblue              #66aabc

	brick red                  brickred                #79443b

	bright chalcedony yellow   brightchalcedonyyellow  #bdda57

	bright chalcedony yellow   brightchalcedonyyellow  #c9dc89

	bright green-yellow        brightgreenyellow       #8db600

	brownish drab              brownishdrab            #977f73

	brownish drab              brownishdrab            #958070

	brownish olive             brownisholive           #6c541e

	brownish vinaceous         brownishvinaceous       #ad8884

	brussels brown             brusselsbrown           #6f4e37

	buckthorn brown            buckthornbrown          #996515

	buff-pink or buff pink     buffpinkorbuffpink      #d9a6a9

	buffy brown                buffybrown              #c19a6b

	buffy brown                buffybrown              #826644

	buffy citrine              buffycitrine            #967117

	buff-yellow or buff yellow buffyelloworbuffyellow  #f8de7e

	buff-yellow or buff yellow buffyelloworbuffyellow  #c9ae5d

	buffy olive                buffyolive              #867e36

	burn blue                  burnblue                #a1caf1

	burn blue                  burnblue                #70a3cc

	burn blue                  burnblue                #b3bce2

	burn blue                  burnblue                #8791bf

	burnt lake                 burntlake               #5d3954

	burnt sienna               burntsienna             #882d17

	burnt umber                burntumber              #674c47

	cacao brown                cacaobrown              #a87c6d

	cadet blue                 cadetblue               #0067a5

	cadet blue                 cadetblue               #436b95

	cadet gray                 cadetgray               #91a3b0

	cadmium orange             cadmiumorange           #f38400

	cadmium orange             cadmiumorange           #ed872d

	cadmium yellow             cadmiumyellow           #ed872d

	cadmium yellow             cadmiumyellow           #eaa221

	calamine blue              calamineblue            #9cd1dc

	calamine blue              calamineblue            #66aabc

	calla green                callagreen              #665d1e

	calliste green             callistegreen           #7e9f2e

	cameo brown                cameobrown              #79443b

	cameo brown                cameobrown              #674c47

	cameo pink                 cameopink               #ffc8d6

	cameo pink                 cameopink               #efbbcc

	capri blue                 capriblue               #367588

	capucine buff              capucinebuff            #fbc97f

	capucine buff              capucinebuff            #e3a857

	capucine buff              capucinebuff            #fad6a5

	capucine orange            capucineorange          #d99058

	capucine yellow            capucineyellow          #eaa221

	carmine                    carmine                 #be0032

	carnelian red              carnelianred            #d9603b

	carob brown                carobbrown              #3e1d1e

	carrot red                 carrotred               #cb6d51

	cartridge buff             cartridgebuff           #f3e5ab

	castor gray                castorgray              #7d8984

	castor gray                castorgray              #848482

	cedar green                cedargreen              #8a9a5b

	cedar green                cedargreen              #4a5d23

	celandine green            celandinegreen          #8da399

	cendre blue                cendreblue              #239eba

	cendre green               cendregreen             #83d37d

	cerulean blue              ceruleanblue            #0067a5

	chaetura black             chaeturablack           #483c32

	chaetura drab              chaeturadrab            #5b504f

	chalcedony yellow          chalcedonyyellow        #eae679

	chamois                    chamois                 #c9ae5d

	chapman's blue             chapmansblue            #436b95

	chartreuse yellow          chartreuseyellow        #eae679

	chatenay pink              chatenaypink            #dea5a4

	chatenay pink              chatenaypink            #d9a6a9

	chessylite blue            chessyliteblue          #367588

	chessylite blue            chessyliteblue          #004958

	chessylite blue            chessyliteblue          #436b95

	chessylite blue            chessyliteblue          #536878

	chestnut                   chestnut                #79443b

	chestnut-brown             chestnutbrown           #79443b

	chestnut-brown             chestnutbrown           #674c47

	chicory blue               chicoryblue             #8791bf

	chicory blue               chicoryblue             #8c82b6

	china blue                 chinablue               #436b95

	chinese violet             chineseviolet           #86608e

	chocolate                  chocolate               #674c47

	chromium green             chromiumgreen           #8a9a5b

	chrysolite green           chrysolitegreen         #8a9a5b

	chrysopraise green         chrysopraisegreen       #93c592

	cinereous                  cinereous               #b4bcc0

	cinereous                  cinereous               #81878b

	cinnamon                   cinnamon                #a67b5b

	cinnamon                   cinnamon                #be8a3d

	cinnamon-brown             cinnamonbrown           #6f4e37

	cinnamon-buff              cinnamonbuff            #e3a857

	cinnamon-buff              cinnamonbuff            #c19a6b

	cinnamon-buff              cinnamonbuff            #c9ae5d

	cinnamon-drab              cinnamondrab            #977f73

	cinnamon-drab              cinnamondrab            #958070

	cinnamon-rufous            cinnamonrufous          #ae6938

	citrine                    citrine                 #867e36

	citrine                    citrine                 #665d1e

	citrine-drab               citrinedrab             #867e36

	citron green               citrongreen             #b9b459

	citron green               citrongreen             #b9b57d

	citron yellow              citronyellow            #e9e450

	citron yellow              citronyellow            #beb72e

	citron yellow              citronyellow            #eae679

	citron yellow              citronyellow            #b9b459

	civette green              civettegreen            #679267

	claret brown               claretbrown             #79443b

	clay color                 claycolor               #996515

	clay color                 claycolor               #826644

	clay color                 claycolor               #967117

	clear cadet blue           clearcadetblue          #4997d0

	clear cadet blue           clearcadetblue          #0067a5

	clear dull green-yellow    cleardullgreenyellow    #bdda57

	clear dull green-yellow    cleardullgreenyellow    #7e9f2e

	clear dull green-yellow    cleardullgreenyellow    #c9dc89

	clear dull green-yellow    cleardullgreenyellow    #8a9a5b

	clear fluorite green       clearfluoritegreen      #93c592

	clear green-blue gray      cleargreenbluegray      #91a3b0

	clear green-blue gray      cleargreenbluegray      #81878b

	clear payne's gray         clearpaynesgray         #536878

	clear payne's gray         clearpaynesgray         #81878b

	clear windsor blue         clearwindsorblue        #8791bf

	clear yellow-green         clearyellowgreen        #bdda57

	clove brown                clovebrown              #483c32

	cobalt green               cobaltgreen             #6aab8e

	colonial buff              colonialbuff            #f8de7e

	columbia blue              columbiablue            #70a3cc

	columbia blue              columbiablue            #436b95

	commelina blue             commelinablue           #30267a

	congo pink                 congopink               #f88379

	congo pink                 congopink               #d9a6a9

	coral pink                 coralpink               #f88379

	coral red                  coralred                #cb6d51

	corinthian pink            corinthianpink          #dea5a4

	corinthian purple          corinthianpurple        #673147

	corinthian red             corinthianred           #905d5d

	cornflower blue            cornflowerblue          #30267a

	corydalis green            corydalisgreen          #c9dc89

	corydalis green            corydalisgreen          #dadfb7

	cossack green              cossackgreen            #4a5d23

	cossack green              cossackgreen            #515744

	cosse green                cossegreen              #7e9f2e

	cotinga purple             cotingapurple           #602f6b

	cotinga purple             cotingapurple           #86608e

	courge green               courgegreen             #8a9a5b

	court gray                 courtgray               #c7e6d7

	court grey                 courtgrey               #c7e6d7

	cream buff                 creambuff               #f8de7e

	cream buff                 creambuff               #c9ae5d

	cream color                creamcolor              #f8de7e

	cress green                cressgreen              #8a9a5b

	cress green                cressgreen              #4a5d23

	cyanine blue               cyanineblue             #4e5180

	dahlia carmine             dahliacarmine           #673147

	dahlia purple              dahliapurple            #673147

	daphne pink                daphnepink              #de6fa1

	daphne pink                daphnepink              #c17e91

	daphne red                 daphnered               #a8516e

	dark american green        darkamericangreen       #355e3b

	dark american green        darkamericangreen       #5e716a

	dark american green        darkamericangreen       #3a4b47

	dark aniline blue          darkanilineblue         #36454f

	dark aniline blue          darkanilineblue         #50404d

	dark anthracene violet     darkanthraceneviolet    #563c5c

	dark anthracene violet     darkanthraceneviolet    #50404d

	dark bluish glaucous       darkbluishglaucous      #6aab8e

	dark bluish glaucous       darkbluishglaucous      #8da399

	dark bluish gray-green     darkbluishgraygreen     #5e716a

	dark bluish gray-green     darkbluishgraygreen     #3a4b47

	dark bluish gray-green     darkbluishgraygreen     #317873

	dark bluish gray-green     darkbluishgraygreen     #004b49

	dark bluish violet         darkbluishviolet        #604e81

	dark bluish violet         darkbluishviolet        #2f2140

	dark bluish violet         darkbluishviolet        #554c69

	dark cadet blue            darkcadetblue           #436b95

	dark cadet blue            darkcadetblue           #4c516d

	dark chessylite blue       darkchessyliteblue      #004958

	dark chessylite blue       darkchessyliteblue      #436b95

	dark chessylite blue       darkchessyliteblue      #00304e

	dark chessylite blue       darkchessyliteblue      #536878

	dark cinnabar green        darkcinnabargreen       #317873

	dark citrine               darkcitrine             #665d1e

	dark corinthian purple     darkcorinthianpurple    #673147

	dark cress green           darkcressgreen          #4a5d23

	dark delft blue            darkdelftblue           #00304e

	dark delft blue            darkdelftblue           #36454f

	dark diva blue             darkdivablue            #545aa7

	dark diva blue             darkdivablue            #4e5180

	dark dull bluish violet    darkdullbluishviolet    #604e81

	dark dull bluish violet    darkdullbluishviolet    #554c69

	dark dull bluish violet    darkdullbluishviolet    #554c69

	dark dull bluish violet    darkdullbluishviolet    #563c5c

	dark dull violet-blue      darkdullvioletblue      #4c516d

	dark dull violet-blue      darkdullvioletblue      #554c69

	dark dull violet-blue      darkdullvioletblue      #554c69

	dark dull yellow-green     darkdullyellowgreen     #4a5d23

	dark dull yellow-green     darkdullyellowgreen     #515744

	dark glaucous-gray         darkglaucousgray        #5e716a

	dark grayish blue-green    darkgrayishbluegreen    #4e5755

	dark grayish blue-green    darkgrayishbluegreen    #555555

	dark grayish blue-violet   darkgrayishblueviolet   #554c69

	dark grayish brown         darkgrayishbrown        #905d5d

	dark grayish brown         darkgrayishbrown        #5c504f

	dark grayish lavender      darkgrayishlavender     #9690ab

	dark grayish olive         darkgrayisholive        #57554c

	dark green                 darkgreen               #355e3b

	dark green                 darkgreen               #5e716a

	dark green                 darkgreen               #3a4b47

	dark green-blue gray       darkgreenbluegray       #536878

	dark green-blue slate      darkgreenblueslate      #00304e

	dark green-blue slate      darkgreenblueslate      #36454f

	dark greenish glaucous     darkgreenishglaucous    #93c592

	dark greenish glaucous     darkgreenishglaucous    #679267

	dark greenish olive        darkgreenisholive       #665d1e

	dark greenish olive        darkgreenisholive       #5b5842

	dark gull gray             darkgullgray            #555555

	dark heliotrope gray       darkheliotropegray      #796878

	dark heliotrope slate      darkheliotropeslate     #50404d

	dark heliotrope slate      darkheliotropeslate     #5d555b

	dark hyssop violet         darkhyssopviolet        #563c5c

	dark hyssop violet         darkhyssopviolet        #796878

	dark indian red            darkindianred           #905d5d

	dark indian red            darkindianred           #543d3f

	dark ivy green             darkivygreen            #515744

	dark lavender              darklavender            #b695c0

	dark livid brown           darklividbrown          #905d5d

	dark livid purple          darklividpurple         #50404d

	dark livid purple          darklividpurple         #673147

	dark madder blue           darkmadderblue          #536878

	dark madder blue           darkmadderblue          #4c516d

	dark madder violet         darkmadderviolet        #563c5c

	dark madder violet         darkmadderviolet        #50404d

	dark maroon purple         darkmaroonpurple        #5d3954

	dark medici blue           darkmediciblue          #51585e

	dark mineral red           darkmineralred          #905d5d

	dark mineral red           darkmineralred          #543d3f

	dark mouse gray            darkmousegray           #5b504f

	dark mouse gray            darkmousegray           #555555

	dark naphthalene violet    darknaphthaleneviolet   #50404d

	dark naphthalene violet    darknaphthaleneviolet   #5d3954

	dark neutral gray          darkneutralgray         #555555

	dark nigrosin violet       darknigrosinviolet      #563c5c

	dark olive                 darkolive               #5b5842

	dark olive-buff            darkolivebuff           #a18f60

	dark olive-gray            darkolivegray           #5b5842

	dark olive-gray            darkolivegray           #57554c

	dark orient blue           darkorientblue          #536878

	dark payne's gray          darkpaynesgray          #36454f

	dark payne's gray          darkpaynesgray          #51585e

	dark perilla purple        darkperillapurple       #673147

	dark plumbago blue         darkplumbagoblue        #9690ab

	dark plumbago blue         darkplumbagoblue        #aa98a9

	dark plumbago gray         darkplumbagogray        #796878

	dark plumbago gray         darkplumbagogray        #8b8589

	dark plumbago gray         darkplumbagogray        #5d555b

	dark plumbago slate        darkplumbagoslate       #5d555b

	dark plumbeous             darkplumbeous           #51585e

	dark porcelain green       darkporcelaingreen      #5e716a

	dark purple-drab           darkpurpledrab          #796878

	dark purplish gray         darkpurplishgray        #5d555b

	dark purplish gray         darkpurplishgray        #555555

	dark quaker drab           darkquakerdrab          #5d555b

	dark russian green         darkrussiangreen        #5e716a

	dark russian green         darkrussiangreen        #3a4b47

	dark slate-purple          darkslatepurple         #5d3954

	dark slate-violet          darkslateviolet         #5d555b

	dark slate-violet          darkslateviolet         #796878

	dark soft blue-violet      darksoftblueviolet      #4e5180

	dark soft blue-violet      darksoftblueviolet      #604e81

	dark soft bluish violet    darksoftbluishviolet    #604e81

	dark sulphate green        darksulphategreen       #317873

	dark terre verte           darkterreverte          #5e716a

	dark tyrian blue           darktyrianblue          #536878

	dark tyrian blue           darktyrianblue          #4c516d

	dark varley's gray         darkvarleysgray         #5d555b

	dark vinaceous             darkvinaceous           #905d5d

	dark vinaceous-brown       darkvinaceousbrown      #905d5d

	dark vinaceous-drab        darkvinaceousdrab       #905d5d

	dark vinaceous-drab        darkvinaceousdrab       #5c504f

	dark vinaceous-gray        darkvinaceousgray       #796878

	dark vinaceous-purple      darkvinaceouspurple     #673147

	dark violet                darkviolet              #604e81

	dark violet-gray           darkvioletgray          #5d555b

	dark violet-gray           darkvioletgray          #555555

	dark violet-slate          darkvioletslate         #536878

	dark viridian green        darkviridiangreen       #3b7861

	dark yellowish green       darkyellowishgreen      #355e3b

	dark yvette violet         darkyvetteviolet        #50404d

	dark zinc green            darkzincgreen           #3b7861

	dauphin's violet           dauphinsviolet          #604e81

	dawn gray                  dawngray                #b4bcc0

	dawn gray                  dawngray                #b9b8b5

	deep aniline lilac         deepanilinelilac        #4997d0

	deep blue-violet           deepblueviolet          #30267a

	deep blue-violet           deepblueviolet          #9065ca

	deep bluish glaucous       deepbluishglaucous      #96ded1

	deep bluish gray-green     deepbluishgraygreen     #5e716a

	deep bluish gray-green     deepbluishgraygreen     #317873

	deep brownish drab         deepbrownishdrab        #635147

	deep brownish vinaceous    deepbrownishvinaceous   #905d5d

	deep cadet blue            deepcadetblue           #436b95

	deep chicory blue          deepchicoryblue         #7e73b8

	deep chrome                deepchrome              #eaa221

	deep chrysolite green      deepchrysolitegreen     #8a9a5b

	deep colonial buff         deepcolonialbuff        #c9ae5d

	deep corinthian red        deepcorinthianred       #905d5d

	deep delft blue            deepdelftblue           #536878

	deep dull bluish violet    deepdullbluishviolet    #604e97

	deep dull bluish violet    deepdullbluishviolet    #604e81

	deep dull bluish violet    deepdullbluishviolet    #8c92ac

	deep dull bluish violet    deepdullbluishviolet    #4c516d

	deep dull bluish violet    deepdullbluishviolet    #9690ab

	deep dull bluish violet    deepdullbluishviolet    #554c69

	deep dull bluish violet    deepdullbluishviolet    #604e81

	deep dull lavender         deepdulllavender        #aa98a9

	deep dull violaceous blue  deepdullviolaceousblue  #4e5180

	deep dull violet-blue      deepdullvioletblue      #604e81

	deep dutch blue            deepdutchblue           #536878

	deep dutch blue            deepdutchblue           #8c92ac

	deep glaucous-gray         deepglaucousgray        #81878b

	deep glaucous-green        deepglaucousgreen       #6aab8e

	deep grape green           deepgrapegreen          #8a9a5b

	deep grape green           deepgrapegreen          #8f9779

	deep grayish blue-green    deepgrayishbluegreen    #5e716a

	deep grayish lavender      deepgrayishlavender     #9690ab

	deep grayish lavender      deepgrayishlavender     #aa98a9

	deep grayish olive         deepgrayisholive        #57554c

	deep green-blue gray       deepgreenbluegray       #536878

	deep greenish glaucous     deepgreenishglaucous    #8ed1b2

	deep gull gray             deepgullgray            #848482

	deep heliotrope gray       deepheliotropegray      #796878

	deep hyssop violet         deephyssopviolet        #86608e

	deep lavender              deeplavender            #8c82b6

	deep lavender-blue         deeplavenderblue        #6c79b8

	deep lichen green          deeplichengreen         #93c592

	deep livid brown           deeplividbrown          #905d5d

	deep livid purple          deeplividpurple         #673147

	deep madder blue           deepmadderblue          #4c516d

	deep malachite green       deepmalachitegreen      #679267

	deep medici blue           deepmediciblue          #81878b

	deep mouse gray            deepmousegray           #5b504f

	deep mouse gray            deepmousegray           #848482

	deep mouse gray            deepmousegray           #555555

	deep neutral gray          deepneutralgray         #555555

	deep olive                 deepolive               #665d1e

	deep olive                 deepolive               #5b5842

	deep olive-buff            deepolivebuff           #c2b280

	deep olive-gray            deepolivegray           #8a8776

	deep orient blue           deeporientblue          #536878

	deep payne's gray          deeppaynesgray          #536878

	deep plumbago blue         deepplumbagoblue        #aa98a9

	deep plumbago gray         deepplumbagogray        #8b8589

	deep plumbeous             deepplumbeous           #536878

	deep plumbeous             deepplumbeous           #81878b

	deep purplish gray         deeppurplishgray        #5d555b

	deep purplish vinaceous    deeppurplishvinaceous   #915f6d

	deep quaker drab           deepquakerdrab          #5d555b

	deep rose pink             deeprosepink            #e68fac

	deep sea-foam green        deepseafoamgreen        #c9dc89

	deep slate-blue            deepslateblue           #536878

	deep slate-green           deepslategreen          #4e5755

	deep slate-olive           deepslateolive          #4e5755

	deep slate-violet          deepslateviolet         #796878

	deep slaty brown           deepslatybrown          #50404d

	deep soft blue-violet      deepsoftblueviolet      #4e5180

	deep soft blue-violet      deepsoftblueviolet      #604e81

	deep soft bluish violet    deepsoftbluishviolet    #604e97

	deep turtle green          deepturtlegreen         #44944a

	deep turtle green          deepturtlegreen         #679267

	deep varley's gray         deepvarleysgray         #796878

	deep varley's gray         deepvarleysgray         #5d555b

	deep vinaceous             deepvinaceous           #c08081

	deep vinaceous-gray        deepvinaceousgray       #796878

	deep vinaceous-lavender    deepvinaceouslavender   #aa8a9e

	deep vinaceous-lavender    deepvinaceouslavender   #c3a6b1

	deep violet-gray           deepvioletgray          #8b8589

	deep violet-gray           deepvioletgray          #5d555b

	deep violet-gray           deepvioletgray          #848482

	deep violet-gray           deepvioletgray          #555555

	deep violet-plumbeous      deepvioletplumbeous     #4c516d

	deep wedgewood blue        deepwedgewoodblue       #6c79b8

	delft blue                 delftblue               #536878

	diamin-azo blue            diaminazoblue           #554c69

	diamin-azo blue            diaminazoblue           #50404d

	diamine brown              diaminebrown            #674c47

	diamine green              diaminegreen            #3b7861

	diamine green              diaminegreen            #1b4d3e

	diva blue                  divablue                #6c79b8

	diva blue                  divablue                #545aa7

	drab                       drab                    #c19a6b

	drab                       drab                    #826644

	drab                       drab                    #ae9b82

	drab                       drab                    #7e6d5a

	drab-gray                  drabgray                #ae9b82

	drab-gray                  drabgray                #bfb8a5

	dragons-blood red          dragonsbloodred         #9e4732

	dresden brown              dresdenbrown            #826644

	duck green                 duckgreen               #3a4b47

	dull blackish green        dullblackishgreen       #3a4b47

	dull blue-green black      dullbluegreenblack      #36454f

	dull blue-green black      dullbluegreenblack      #202830

	dull blue-green black      dullbluegreenblack      #202428

	dull blue-violet           dullblueviolet          #545aa7

	dull blue-violet           dullblueviolet          #7e73b8

	dull blue-violet           dullblueviolet          #604e97

	dull blue-violet           dullblueviolet          #604e81

	dull bluish violet         dullbluishviolet        #7e73b8

	dull bluish violet         dullbluishviolet        #604e97

	dull bluish violet         dullbluishviolet        #8c82b6

	dull bluish violet         dullbluishviolet        #604e81

	dull bluish violet         dullbluishviolet        #604e97

	dull bluish violet         dullbluishviolet        #604e81

	dull citrine               dullcitrine             #867e36

	dull citrine               dullcitrine             #665d1e

	dull dark purple           dulldarkpurple          #9e4f88

	dull dark purple           dulldarkpurple          #702963

	dull dark purple           dulldarkpurple          #915c83

	dull dusky purple          dullduskypurple         #50404d

	dull dusky purple          dullduskypurple         #5d3954

	dull greenish black        dullgreenishblack       #4e5755

	dull greenish black        dullgreenishblack       #555555

	dull greenish black        dullgreenishblack       #4e5755

	dull greenish black        dullgreenishblack       #555555

	dull green-yellow          dullgreenyellow         #7e9f2e

	dull indian purple         dullindianpurple        #915f6d

	dull lavender              dulllavender            #c3a6b1

	dull magenta purple        dullmagentapurple       #9e4f88

	dull opaline green         dullopalinegreen        #c7e6d7

	dull purplish black        dullpurplishblack       #5d555b

	dull violaceous blue       dullviolaceousblue      #545aa7

	dull violet-black          dullvioletblack         #5d555b

	dull violet-black          dullvioletblack         #36454f

	dull violet-black          dullvioletblack         #51585e

	dull violet-black          dullvioletblack         #50404d

	dull violet-black          dullvioletblack         #5d555b

	dull violet-black          dullvioletblack         #555555

	dull violet-black          dullvioletblack         #222222

	dull violet-blue           dullvioletblue          #545aa7

	dull violet-blue           dullvioletblue          #604e81

	dusky auricula purple      duskyauriculapurple     #5d3954

	dusky blue                 duskyblue               #4c516d

	dusky blue-green           duskybluegreen          #3a4b47

	dusky blue-violet          duskyblueviolet         #554c69

	dusky blue-violet          duskyblueviolet         #2f2140

	dusky blue-violet          duskyblueviolet         #50404d

	dusky bluish green         duskybluishgreen        #4e5755

	dusky brown                duskybrown              #50404d

	dusky brown                duskybrown              #5d555b

	dusky drab                 duskydrab               #5b504f

	dusky dull bluish green    duskydullbluishgreen    #3a4b47

	dusky dull green           duskydullgreen          #3a4b47

	dusky dull green           duskydullgreen          #4e5755

	dusky dull violet          duskydullviolet         #563c5c

	dusky dull violet          duskydullviolet         #86608e

	dusky dull violet          duskydullviolet         #563c5c

	dusky dull violet          duskydullviolet         #796878

	dusky dull violet-blue     duskydullvioletblue     #554c69

	dusky green                duskygreen              #5e716a

	dusky green                duskygreen              #3a4b47

	dusky green-blue           duskygreenblue          #004b49

	dusky green-blue           duskygreenblue          #004958

	dusky green-blue           duskygreenblue          #536878

	dusky green-blue           duskygreenblue          #536878

	dusky green-blue           duskygreenblue          #36454f

	dusky green-gray           duskygreengray          #555555

	dusky greenish blue        duskygreenishblue       #00416a

	dusky greenish blue        duskygreenishblue       #436b95

	dusky greenish blue        duskygreenishblue       #00304e

	dusky neutral gray         duskyneutralgray        #555555

	dusky olive-green          duskyolivegreen         #515744

	dusky orient blue          duskyorientblue         #536878

	dusky purplish gray        duskypurplishgray       #50404d

	dusky purplish gray        duskypurplishgray       #5d555b

	dusky purplish gray        duskypurplishgray       #555555

	dusky purplish gray        duskypurplishgray       #222222

	dusky slate-blue           duskyslateblue          #36454f

	dusky slate-violet         duskyslateviolet        #50404d

	dusky slate-violet         duskyslateviolet        #5d555b

	dusky violet               duskyviolet             #50404d

	dusky violet-blue          duskyvioletblue         #4c516d

	dusky violet-blue          duskyvioletblue         #554c69

	dusky violet-blue          duskyvioletblue         #36454f

	dusky yellowish green      duskyyellowishgreen     #515744

	dutch blue                 dutchblue               #91a3b0

	dutch blue                 dutchblue               #8c92ac

	ecru-drab                  ecrudrab                #c2ac99

	ecru-olive                 ecruolive               #ab9144

	ecru-olive                 ecruolive               #a18f60

	elm green                  elmgreen                #4a5d23

	elm green                  elmgreen                #515744

	emerald green              emeraldgreen            #83d37d

	empire green               empiregreen             #5e716a

	empire green               empiregreen             #3a4b47

	empire yellow              empireyellow            #fada5e

	endive blue                endiveblue              #8791bf

	endive blue                endiveblue              #8c92ac

	english red                englishred              #9e4732

	eosine pink                eosinepink              #e4717a

	ethyl green                ethylgreen              #007a74

	ethyl green                ethylgreen              #317873

	eton blue                  etonblue                #436b95

	etruscan red               etruscanred             #905d5d

	eugenia red                eugeniared              #ab4e52

	eupatorium purple          eupatoriumpurple        #b784a7

	eupatorium purple          eupatoriumpurple        #915c83

	fawn color                 fawncolor               #977f73

	fawn color                 fawncolor               #a67b5b

	fawn color                 fawncolor               #958070

	ferruginous                ferruginous             #cb6d51

	ferruginous                ferruginous             #9e4732

	flame scarlet              flamescarlet            #e25822

	flax-flower blue           flaxflowerblue          #6c79b8

	flesh color                fleshcolor              #dea5a4

	flesh-ochre                fleshochre              #d99058

	flesh pink                 fleshpink               #f4c2c2

	flesh pink                 fleshpink               #d9a6a9

	fluorite green             fluoritegreen           #679267

	fluorite violet            fluoriteviolet          #563c5c

	forest green               forestgreen             #8a9a5b

	forest green               forestgreen             #4a5d23

	forget-me-not blue         forgetmenotblue         #6c79b8

	french gray                frenchgray              #91a3b0

	french gray                frenchgray              #b4bcc0

	french green               frenchgreen             #3b7861

	fuscous                    fuscous                 #5b504f

	fuscous-black              fuscousblack            #5b504f

	fuscous-black              fuscousblack            #555555

	garnet brown               garnetbrown             #841b2d

	gendarme blue              gendarmeblue            #436b95

	gentian blue               gentianblue             #545aa7

	geranium pink              geraniumpink            #e4717a

	geranium pink              geraniumpink            #e66761

	glass green                glassgreen              #c9dc89

	glaucous                   glaucous                #dadfb7

	glaucous-blue              glaucousblue            #66aabc

	glaucous-gray              glaucousgray            #b4bcc0

	glaucous-green             glaucousgreen           #8da399

	gnaphalium green           gnaphaliumgreen         #c7e6d7

	gnaphalium green           gnaphaliumgreen         #8da399

	grass green                grassgreen              #8a9a5b

	grayish blue-green         grayishbluegreen        #5e716a

	grayish blue-violet        grayishblueviolet       #545aa7

	grayish blue-violet        grayishblueviolet       #4e5180

	grayish blue-violet        grayishblueviolet       #604e97

	grayish blue-violet        grayishblueviolet       #604e81

	grayish blue-violet        grayishblueviolet       #8791bf

	grayish lavender           grayishlavender         #aa98a9

	grayish olive              grayisholive            #8c8767

	grayish violaceous blue    grayishviolaceousblue   #8791bf

	grayish violet-blue        grayishvioletblue       #8c82b6

	grayish violet-blue        grayishvioletblue       #604e81

	green-blue slate           greenblueslate          #536878

	greenish glaucous          greenishglaucous        #c7e6d7

	greenish glaucous-blue     greenishglaucousblue    #66ada4

	greenish slate-black       greenishslateblack      #3a4b47

	greenish slate-black       greenishslateblack      #4e5755

	greenish slate-black       greenishslateblack      #555555

	greenish slate-black       greenishslateblack      #222222

	greenish yellow            greenishyellow          #beb72e

	green-yellow               greenyellow             #e9e450

	grenadine                  grenadine               #d9603b

	grenadine pink             grenadinepink           #f88379

	grenadine red              grenadinered            #e25822

	guinea green               guineagreen             #007a74

	gull gray                  gullgray                #848482

	haematite red              haematitered            #674c47

	haematoxylin violet        haematoxylinviolet      #86608e

	hair brown                 hairbrown               #5b504f

	hathi gray                 hathigray               #b2beb5

	hathi gray                 hathigray               #7d8984

	hathi gray                 hathigray               #b9b8b5

	hathi gray                 hathigray               #848482

	hay's blue                 haysblue                #545aa7

	hay's brown                haysbrown               #905d5d

	hay's brown                haysbrown               #674c47

	hay's green                haysgreen               #679267

	hay's green                haysgreen               #355e3b

	hay's lilac                hayslilac               #b695c0

	hay's maroon               haysmaroon              #543d3f

	hay's russet               haysrusset              #79443b

	hazel                      hazel                   #80461b

	heliotrope gray            heliotropegray          #aa98a9

	heliotrope gray            heliotropegray          #796878

	heliotrope slate           heliotropeslate         #796878

	hellebore red              helleborered            #a8516e

	helvetia blue              helvetiablue            #545aa7

	hermosa pink               hermosapink             #ea9399

	hessian brown              hessianbrown            #674c47

	hessian brown              hessianbrown            #43302e

	honey yellow               honeyyellow             #c9ae5d

	honey yellow               honeyyellow             #ab9144

	hortense blue              hortenseblue            #436b95

	hortense violet            hortenseviolet          #d399e6

	hyacinth blue              hyacinthblue            #272458

	hyacinth blue              hyacinthblue            #604e81

	hyacinth violet            hyacinthviolet          #602f6b

	hydrangea pink             hydrangeapink           #f4c2c2

	hydrangea pink             hydrangeapink           #d9a6a9

	hydrangea red              hydrangeared            #905d5d

	hyssop violet              hyssopviolet            #86608e

	indian lake                indianlake              #a8516e

	indian lake                indianlake              #915f6d

	indian purple              indianpurple            #50404d

	indian purple              indianpurple            #673147

	indian red                 indianred               #905d5d

	indigo blue                indigoblue              #00304e

	indigo blue                indigoblue              #536878

	indigo blue                indigoblue              #36454f

	indigo blue                indigoblue              #4c516d

	indulin blue               indulinblue             #4c516d

	invisible green            invisiblegreen          #3a4b47

	invisible green            invisiblegreen          #004b49

	iron gray                  irongray                #57554c

	isabella color             isabellacolor           #a18f60

	italian blue               italianblue             #007791

	ivory yellow               ivoryyellow             #f3e5ab

	ivy green                  ivygreen                #665d1e

	ivy green                  ivygreen                #5b5842

	jade green                 jadegreen               #4a5d23

	jade green                 jadegreen               #515744

	japan rose                 japanrose               #d9a6a9

	jasper green               jaspergreen             #3b7861

	jasper pink                jasperpink              #e4717a

	jasper pink                jasperpink              #f88379

	jasper pink                jasperpink              #e66761

	jasper red                 jasperred               #ab4e52

	javel green                javelgreen              #b9b459

	jay blue                   jayblue                 #436b95

	jay blue                   jayblue                 #8791bf

	jouvence blue              jouvenceblue            #317873

	jouvence blue              jouvenceblue            #367588

	kaiser brown               kaiserbrown             #80461b

	kildare green              kildaregreen            #8a9a5b

	killarney green            killarneygreen          #679267

	king's blue                kingsblue               #70a3cc

	kronberg's green           kronbergsgreen          #867e36

	kronberg's green           kronbergsgreen          #665d1e

	laelia pink                laeliapink              #b784a7

	laelia pink                laeliapink              #de6fa1

	la france pink             lafrancepink            #ffb5ba

	lavender                   lavender                #dcd0ff

	lavender                   lavender                #c4c3dd

	lavender-blue              lavenderblue            #6c79b8

	lavender-blue              lavenderblue            #8791bf

	lavender-gray              lavendergray            #c0c8e1

	lavender-gray              lavendergray            #8c92ac

	lavender-gray              lavendergray            #c4c3dd

	lavender-gray              lavendergray            #9690ab

	lavender-violet            lavenderviolet          #7e73b8

	leaf green                 leafgreen               #515744

	leitch's blue              leitchsblue             #0067a5

	lemon chrome               lemonchrome             #d4af37

	lemon yellow               lemonyellow             #e9e450

	lemon yellow               lemonyellow             #beb72e

	lettuce green              lettucegreen            #8a9a5b

	lichen green               lichengreen             #c7e6d7

	light alice blue           lightaliceblue          #70a3cc

	light alice blue           lightaliceblue          #91a3b0

	light amparo blue          lightamparoblue         #4997d0

	light amparo purple        lightamparopurple       #d399e6

	light bice green           lightbicegreen          #8a9a5b

	light blue-green           lightbluegreen          #3eb489

	light blue-green           lightbluegreen          #6aab8e

	light blue-violet          lightblueviolet         #30267a

	light bluish violet        lightbluishviolet       #6c79b8

	light bluish violet        lightbluishviolet       #545aa7

	light bluish violet        lightbluishviolet       #7e73b8

	light brownish drab        lightbrownishdrab       #958070

	light brownish olive       lightbrownisholive      #967117

	light brownish vinaceous   lightbrownishvinaceous  #c4aead

	light buff                 lightbuff               #fad6a5

	light buff                 lightbuff               #f3e5ab

	light cadet blue           lightcadetblue          #4997d0

	light cadmium              lightcadmium            #f3c300

	light campanula blue       lightcampanulablue      #6c79b8

	light celandine green      lightcelandinegreen     #b2beb5

	light celandine green      lightcelandinegreen     #7d8984

	light cendre green         lightcendregreen        #83d37d

	light cendre green         lightcendregreen        #93c592

	light cerulean blue        lightceruleanblue       #70a3cc

	light chalcedony yellow    lightchalcedonyyellow   #eae679

	light chicory blue         lightchicoryblue        #8c82b6

	light cinnamon-drab        lightcinnamondrab       #977f73

	light cinnamon-drab        lightcinnamondrab       #958070

	light columbia blue        lightcolumbiablue       #70a3cc

	light congo pink           lightcongopink          #f4c2c2

	light congo pink           lightcongopink          #d9a6a9

	light coral red            lightcoralred           #cb6d51

	light corinthian red       lightcorinthianred      #c08081

	light cress green          lightcressgreen         #8a9a5b

	light danube green         lightdanubegreen        #3b7861

	light drab                 lightdrab               #ae9b82

	light dull bluish violet   lightdullbluishviolet   #8c82b6

	light dull glaucous-blue   lightdullglaucousblue   #66aabc

	light dull glaucous-blue   lightdullglaucousblue   #91a3b0

	light dull green-yellow    lightdullgreenyellow    #c9dc89

	light elm green            lightelmgreen           #8a9a5b

	light fluorite green       lightfluoritegreen      #c9dc89

	light forget-me-not blue   lightforgetmenotblue    #70a3cc

	light glaucous-blue        lightglaucousblue       #9cd1dc

	light grape green          lightgrapegreen         #8a9a5b

	light grayish blue-violet  lightgrayishblueviolet  #70a3cc

	light grayish blue-violet  lightgrayishblueviolet  #8791bf

	light grayish olive        lightgrayisholive       #8c8767

	light grayish olive        lightgrayisholive       #8a8776

	light grayish vinaceous    lightgrayishvinaceous   #c7ada3

	light grayish violet-blue  lightgrayishvioletblue  #6c79b8

	light greenish yellow      lightgreenishyellow     #e9e450

	light greenish yellow      lightgreenishyellow     #beb72e

	light green-yellow         lightgreenyellow        #eae679

	light gull gray            lightgullgray           #b9b8b5

	light heliotrope gray      lightheliotropegray     #aa98a9

	light hellebore green      lighthelleboregreen     #8a9a5b

	light hellebore green      lighthelleboregreen     #8f9779

	light hortense violet      lighthortenseviolet     #d399e6

	light hyssop violet        lighthyssopviolet       #8c82b6

	light jasper red           lightjasperred          #e4717a

	light jasper red           lightjasperred          #ab4e52

	light king's blue          lightkingsblue          #bcd4e6

	light king's blue          lightkingsblue          #91a3b0

	light lavender-blue        lightlavenderblue       #8791bf

	light lavender-violet      lightlavenderviolet     #8c82b6

	light lobelia violet       lightlobeliaviolet      #b695c0

	light lumiere green        lightlumieregreen       #c9dc89

	light mallow purple        lightmallowpurple       #de6fa1

	light mauve                lightmauve              #8c82b6

	light medici blue          lightmediciblue         #536878

	light methyl blue          lightmethylblue         #4997d0

	light methyl blue          lightmethylblue         #0067a5

	light mineral gray         lightmineralgray        #b2beb5

	light mouse gray           lightmousegray          #8e8279

	light neropalin blue       lightneropalinblue      #91a3b0

	light neutral gray         lightneutralgray        #848482

	light niagara green        lightniagaragreen       #66ada4

	light ochraceous-buff      lightochraceousbuff     #e3a857

	light ochraceous-salmon    lightochraceoussalmon   #fad6a5

	light olive-gray           lightolivegray          #bfb8a5

	light orange-yellow        lightorangeyellow       #e3a857

	light orange-yellow        lightorangeyellow       #d4af37

	light oriental green       lightorientalgreen      #93c592

	light paris green          lightparisgreen         #93c592

	light payne's gray         lightpaynesgray         #91a3b0

	light payne's gray         lightpaynesgray         #81878b

	light perilla purple       lightperillapurple      #915c83

	light perilla purple       lightperillapurple      #836479

	light phlox purple         lightphloxpurple        #de6fa1

	light pinkish cinnamon     lightpinkishcinnamon    #f4c2c2

	light pinkish cinnamon     lightpinkishcinnamon    #d9a6a9

	light pinkish lilac        lightpinkishlilac       #efbbcc

	light plumbago gray        lightplumbagogray       #8b8589

	light porcelain green      lightporcelaingreen     #66ada4

	light porcelain green      lightporcelaingreen     #317873

	light purple-drab          lightpurpledrab         #796878

	light purplish gray        lightpurplishgray       #8b8589

	light purplish gray        lightpurplishgray       #848482

	light purplish vinaceous   lightpurplishvinaceous  #dea5a4

	light quaker drab          lightquakerdrab         #8b8589

	light rosolane purple      lightrosolanepurple     #de6fa1

	light russet-vinaceous     lightrussetvinaceous    #a87c6d

	light salmon-orange        lightsalmonorange       #d99058

	light seal brown           lightsealbrown          #5c504f

	light sky blue             lightskyblue            #91a3b0

	light soft blue-violet     lightsoftblueviolet     #6c79b8

	light soft blue-violet     lightsoftblueviolet     #7e73b8

	light squill blue          lightsquillblue         #70a3cc

	light sulphate green       lightsulphategreen      #66ada4

	light terre verte          lightterreverte         #317873

	light turtle green         lightturtlegreen        #c9dc89

	light tyrian blue          lighttyrianblue         #536878

	light tyrian blue          lighttyrianblue         #8c92ac

	light tyrian blue          lighttyrianblue         #4c516d

	light varley's gray        lightvarleysgray        #aa98a9

	light vinaceous-cinnamon   lightvinaceouscinnamon  #fad6a5

	light vinaceous-drab       lightvinaceousdrab      #905d5d

	light vinaceous-fawn       lightvinaceousfawn      #c7ada3

	light vinaceous-fawn       lightvinaceousfawn      #c2ac99

	light vinaceous-gray       lightvinaceousgray      #aa98a9

	light vinaceous-gray       lightvinaceousgray      #bfb9bd

	light vinaceous-gray       lightvinaceousgray      #8b8589

	light vinaceous-gray       lightvinaceousgray      #c3a6b1

	light vinaceous-lilac      lightvinaceouslilac     #b784a7

	light vinaceous-lilac      lightvinaceouslilac     #aa8a9e

	light vinaceous-lilac      lightvinaceouslilac     #d597ae

	light vinaceous-lilac      lightvinaceouslilac     #c3a6b1

	light vinaceous-purple     lightvinaceouspurple    #836479

	light violet               lightviolet             #6c79b8

	light violet               lightviolet             #7e73b8

	light violet-blue          lightvioletblue         #545aa7

	light violet-gray          lightvioletgray         #8b8589

	light violet-gray          lightvioletgray         #848482

	light violet-plumbeous     lightvioletplumbeous    #91a3b0

	light violet-plumbeous     lightvioletplumbeous    #536878

	light violet-plumbeous     lightvioletplumbeous    #8c92ac

	light viridine green       lightviridinegreen      #c9dc89

	light viridine yellow      lightviridineyellow     #eae679

	light windsor blue         lightwindsorblue        #8791bf

	light wistaria blue        lightwistariablue       #8791bf

	light wistaria blue        lightwistariablue       #8c82b6

	light wistaria violet      lightwistariaviolet     #8c82b6

	light yellow-green         lightyellowgreen        #bdda57

	light yellow-green         lightyellowgreen        #c9dc89

	light yellowish olive      lightyellowisholive     #98943e

	lilac                      lilac                   #b695c0

	lilac                      lilac                   #aa98a9

	lilac gray                 lilacgray               #bfb9bd

	lily green                 lilygreen               #7d8984

	lime green                 limegreen               #b9b459

	lime green                 limegreen               #98943e

	lime green                 limegreen               #867e36

	lincoln green              lincolngreen            #515744

	liseran purple             liseranpurple           #9e4f88

	liseran purple             liseranpurple           #b784a7

	liseran purple             liseranpurple           #915c83

	liseran purple             liseranpurple           #de6fa1

	litho purple               lithopurple             #875692

	litho purple               lithopurple             #602f6b

	litho purple               lithopurple             #86608e

	liver brown                liverbrown              #79443b

	livid brown                lividbrown              #905d5d

	livid pink                 lividpink               #f9ccca

	livid pink                 lividpink               #dea5a4

	livid pink                 lividpink               #ead8d7

	livid pink                 lividpink               #c4aead

	livid purple               lividpurple             #915c83

	livid violet               lividviolet             #86608e

	livid violet               lividviolet             #796878

	lobelia violet             lobeliaviolet           #b695c0

	lumiere blue               lumiereblue             #66ada4

	lumiere blue               lumiereblue             #66aabc

	lumiere green              lumieregreen            #c9dc89

	lumiere green              lumieregreen            #8a9a5b

	lyons blue                 lyonsblue               #0067a5

	madder blue                madderblue              #9690ab

	madder blue                madderblue              #796878

	madder brown               madderbrown             #905d5d

	madder violet              madderviolet            #563c5c

	magenta                    magenta                 #9e4f88

	mahogany red               mahoganyred             #79443b

	maize yellow               maizeyellow             #f8de7e

	malachite green            malachitegreen          #93c592

	mallow pink                mallowpink              #e68fac

	mallow purple              mallowpurple            #870074

	mallow purple              mallowpurple            #de6fa1

	manganese violet           manganeseviolet         #875692

	marguerite yellow          margueriteyellow        #f3e5ab

	marine blue                marineblue              #436b95

	maroon                     maroon                  #3e1d1e

	mars brown                 marsbrown               #6f4e37

	mars orange                marsorange              #cb6d51

	mars orange                marsorange              #9e4732

	mars violet                marsviolet              #50404d

	mars yellow                marsyellow              #c98500

	mars yellow                marsyellow              #be8a3d

	martius yellow             martiusyellow           #eae679

	massicot yellow            massicotyellow          #f8de7e

	massicot yellow            massicotyellow          #f3e5ab

	mathew's blue              mathewsblue             #0067a5

	mathew's purple            mathewspurple           #875692

	mathew's purple            mathewspurple           #86608e

	mauve                      mauve                   #7e73b8

	mauvette                   mauvette                #efbbcc

	mazarine blue              mazarineblue            #4997d0

	mazarine blue              mazarineblue            #70a3cc

	meadow green               meadowgreen             #679267

	medal bronze               medalbronze             #6c541e

	medici blue                mediciblue              #81878b

	methyl blue                methylblue              #00a1c2

	methyl green               methylgreen             #007a74

	methyl green               methylgreen             #317873

	microcline green           microclinegreen         #96ded1

	mignonette green           mignonettegreen         #867e36

	mikado brown               mikadobrown             #a67b5b

	mikado orange              mikadoorange            #ed872d

	mineral gray               mineralgray             #8f9779

	mineral green              mineralgreen            #7e9f2e

	mineral red                mineralred              #905d5d

	montpellier green          montpelliergreen        #6aab8e

	montpellier green          montpelliergreen        #3b7861

	morocco red                moroccored              #79443b

	motmot blue                motmotblue              #367588

	motmot green               motmotgreen             #44944a

	motmot green               motmotgreen             #679267

	mouse gray                 mousegray               #8e8279

	mulberry purple            mulberrypurple          #602f6b

	mulberry purple            mulberrypurple          #563c5c

	mummy brown                mummybrown              #4b3621

	munich lake                munichlake              #be0032

	munich lake                munichlake              #bc3f4a

	mustard yellow             mustardyellow           #d4af37

	myrtle green               myrtlegreen             #004b49

	mytho green                mythogreen              #8a9a5b

	mytho green                mythogreen              #8f9779

	naphthalene violet         naphthaleneviolet       #5d3954

	naphthalene yellow         naphthaleneyellow       #eae679

	naphthalene yellow         naphthaleneyellow       #ebe8a4

	naples yellow              naplesyellow            #f8de7e

	natal brown                natalbrown              #635147

	navy blue                  navyblue                #4e5180

	navy blue                  navyblue                #4c516d

	neropalin blue             neropalinblue           #70a3cc

	neutral gray               neutralgray             #848482

	neutral red                neutralred              #673147

	neutral red                neutralred              #915f6d

	neuvider green             neuvidergreen           #93c592

	neva green                 nevagreen               #bdda57

	niagara green              niagaragreen            #8da399

	niagara green              niagaragreen            #66ada4

	nickel green               nickelgreen             #3b7861

	nickel green               nickelgreen             #5e716a

	night green                nightgreen              #8db600

	nigrosin blue              nigrosinblue            #4c516d

	nigrosin violet            nigrosinviolet          #563c5c

	nile blue                  nileblue                #66ada4

	nopal red                  nopalred                #bc3f4a

	ocher red                  ocherred                #905d5d

	ochraceous-buff            ochraceousbuff          #f4c2c2

	ochraceous-buff            ochraceousbuff          #d9a6a9

	ochraceous-orange          ochraceousorange        #eaa221

	ochraceous-orange          ochraceousorange        #c98500

	ochraceous-salmon          ochraceoussalmon        #d99058

	ochraceous-tawny           ochraceoustawny         #be8a3d

	ochraceous-tawny           ochraceoustawny         #996515

	oil green                  oilgreen                #8a9a5b

	old burgundy               oldburgundy             #2e1d21

	old gold                   oldgold                 #ab9144

	old gold                   oldgold                 #967117

	old rose                   oldrose                 #ab4e52

	olive                      olive                   #665d1e

	olive                      olive                   #5b5842

	olive-brown                olivebrown              #7e6d5a

	olive-buff                 olivebuff               #f3e5ab

	olive-buff                 olivebuff               #c2b280

	olive-citrine              olivecitrine            #665d1e

	olive-gray                 olivegray               #8c8767

	olive-gray                 olivegray               #8a8776

	olive green                olivegreen              #665d1e

	olive lake                 olivelake               #a18f60

	olive lake                 olivelake               #867e36

	olive-ocher                oliveocher              #c9ae5d

	olive-ocher                oliveocher              #ab9144

	olive-yellow               oliveyellow             #b9b459

	olivine                    olivine                 #93c592

	olympic blue               olympicblue             #0067a5

	onion-skin pink            onionskinpink           #d9a6a9

	onion-skin pink            onionskinpink           #d99058

	ontario violet             ontarioviolet           #8c82b6

	opaline green              opalinegreen            #c9dc89

	opaline green              opalinegreen            #dadfb7

	orange                     orange                  #ed872d

	orange-buff                orangebuff              #e3a857

	orange chrome              orangechrome            #f38400

	orange-cinnamon            orangecinnamon          #d99058

	orange-citrine             orangecitrine           #967117

	orange-pink                orangepink              #fd943f

	orange-pink                orangepink              #fab57f

	orange-rufous              orangerufous            #be6516

	orange-vinaceous           orangevinaceous         #e4717a

	orange-vinaceous           orangevinaceous         #f88379

	oriental green             orientalgreen           #679267

	orient blue                orientblue              #436b95

	orient blue                orientblue              #536878

	orient pink                orientpink              #f88379

	oural green                ouralgreen              #b6e5af

	oural green                ouralgreen              #93c592

	ox-blood red               oxbloodred              #841b2d

	oxide blue                 oxideblue               #0067a5

	oxide blue                 oxideblue               #436b95

	pale amaranth pink         paleamaranthpink        #e68fac

	pale amparo blue           paleamparoblue          #bcd4e6

	pale amparo blue           paleamparoblue          #91a3b0

	pale amparo purple         paleamparopurple        #b695c0

	pale aniline lilac         paleanilinelilac        #a1caf1

	pale aniline lilac         paleanilinelilac        #70a3cc

	pale blue                  paleblue                #9cd1dc

	pale blue                  paleblue                #bcd4e6

	pale blue-green            palebluegreen           #c7e6d7

	pale blue-violet           paleblueviolet          #6c79b8

	pale bluish lavender       palebluishlavender      #c4c3dd

	pale bluish lavender       palebluishlavender      #d6cadd

	pale bluish violet         palebluishviolet        #6c79b8

	pale brownish drab         palebrownishdrab        #c1b6b3

	pale brownish vinaceous    palebrownishvinaceous   #ead8d7

	pale brownish vinaceous    palebrownishvinaceous   #c4aead

	pale cadet blue            palecadetblue           #70a3cc

	pale cadet blue            palecadetblue           #91a3b0

	pale cadet blue            palecadetblue           #8791bf

	pale cadet blue            palecadetblue           #8c92ac

	pale campanula blue        palecampanulablue       #b3bce2

	pale campanula blue        palecampanulablue       #8791bf

	pale cendre green          palecendregreen         #c9dc89

	pale cendre green          palecendregreen         #dadfb7

	pale cerulean blue         paleceruleanblue        #a1caf1

	pale cerulean blue         paleceruleanblue        #70a3cc

	pale chalcedony yellow     palechalcedonyyellow    #ebe8a4

	pale cinnamon-pink         palecinnamonpink        #f4c2c2

	pale cinnamon-pink         palecinnamonpink        #d9a6a9

	pale cinnamon-pink         palecinnamonpink        #ecd5c5

	pale cinnamon-pink         palecinnamonpink        #c2ac99

	pale congo pink            palecongopink           #f4c2c2

	pale congo pink            palecongopink           #d9a6a9

	pale congo pink            palecongopink           #ecd5c5

	pale congo pink            palecongopink           #c7ada3

	pale drab-gray             paledrabgray            #f2f3f4

	pale drab-gray             paledrabgray            #b9b8b5

	pale dull glaucous-blue    paledullglaucousblue    #bcd4e6

	pale dull glaucous-blue    paledullglaucousblue    #91a3b0

	pale dull green-yellow     paledullgreenyellow     #c9dc89

	pale ecru-drab             paleecrudrab            #c1b6b3

	pale flesh color           palefleshcolor          #f4c2c2

	pale flesh color           palefleshcolor          #d9a6a9

	pale fluorite green        palefluoritegreen       #c7e6d7

	pale glass green           paleglassgreen          #c9dc89

	pale glass green           paleglassgreen          #dadfb7

	pale glaucous-blue         paleglaucousblue        #9cd1dc

	pale glaucous-blue         paleglaucousblue        #bcd4e6

	pale glaucous-green        paleglaucousgreen       #c7e6d7

	pale grayish blue          palegrayishblue         #bcd4e6

	pale grayish blue          palegrayishblue         #c0c8e1

	pale grayish blue-violet   palegrayishblueviolet   #c0c8e1

	pale grayish vinaceous     palegrayishvinaceous    #c1b6b3

	pale grayish vinaceous     palegrayishvinaceous    #ecd5c5

	pale grayish vinaceous     palegrayishvinaceous    #c7ada3

	pale grayish violet-blue   palegrayishvioletblue   #8791bf

	pale green-blue gray       palegreenbluegray       #91a3b0

	pale greenish yellow       palegreenishyellow      #eae679

	pale greenish yellow       palegreenishyellow      #b9b459

	pale green-yellow          palegreenyellow         #eae679

	pale green-yellow          palegreenyellow         #ebe8a4

	pale gull gray             palegullgray            #b9b8b5

	pale hortense violet       palehortenseviolet      #d5badb

	pale hortense violet       palehortenseviolet      #b695c0

	pale king's blue           palekingsblue           #c4c3dd

	pale laelia pink           palelaeliapink          #efbbcc

	pale laelia pink           palelaeliapink          #d597ae

	pale lavender-violet       palelavenderviolet      #b695c0

	pale lavender-violet       palelavenderviolet      #aa98a9

	pale lemon yellow          palelemonyellow         #e9e450

	pale lilac                 palelilac               #e8ccd7

	pale lobelia violet        palelobeliaviolet       #d6cadd

	pale lobelia violet        palelobeliaviolet       #aa98a9

	pale lumiere green         palelumieregreen        #c9dc89

	pale mauve                 palemauve               #d6cadd

	pale mazarine blue         palemazarineblue        #bcd4e6

	pale mazarine blue         palemazarineblue        #91a3b0

	pale mazarine blue         palemazarineblue        #c0c8e1

	pale mazarine blue         palemazarineblue        #8c92ac

	pale medici blue           palemediciblue          #91a3b0

	pale medici blue           palemediciblue          #b4bcc0

	pale medici blue           palemediciblue          #81878b

	pale methyl blue           palemethylblue          #70a3cc

	pale mouse gray            palemousegray           #8e8279

	pale mouse gray            palemousegray           #bfb8a5

	pale neropalin blue        paleneropalinblue       #bcd4e6

	pale neutral gray          paleneutralgray         #848482

	pale niagara green         paleniagaragreen        #c7e6d7

	pale nile blue             palenileblue            #96ded1

	pale ochraceous-buff       paleochraceousbuff      #f4c2c2

	pale ochraceous-salmon     paleochraceoussalmon    #fad6a5

	pale ochraceous-salmon     paleochraceoussalmon    #f3e5ab

	pale olive-buff            paleolivebuff           #f3e5ab

	pale olive-gray            paleolivegray           #f0ead6

	pale olive-gray            paleolivegray           #bfb8a5

	pale olivine               paleolivine             #c7e6d7

	pale orange-yellow         paleorangeyellow        #fbc97f

	pale orange-yellow         paleorangeyellow        #e3a857

	pale orange-yellow         paleorangeyellow        #fada5e

	pale orange-yellow         paleorangeyellow        #d4af37

	pale orange-yellow         paleorangeyellow        #f8de7e

	pale orange-yellow         paleorangeyellow        #c9ae5d

	pale payne's gray          palepaynesgray          #91a3b0

	pale persian lilac         palepersianlilac        #d597ae

	pale pinkish buff          palepinkishbuff         #fad6a5

	pale pinkish cinnamon      palepinkishcinnamon     #fad6a5

	pale purple-drab           palepurpledrab          #c3a6b1

	pale purplish gray         palepurplishgray        #8b8589

	pale purplish vinaceous    palepurplishvinaceous   #f9ccca

	pale purplish vinaceous    palepurplishvinaceous   #dea5a4

	pale quaker drab           palequakerdrab          #bfb9bd

	pale quaker drab           palequakerdrab          #8b8589

	pale rhodonite pink        palerhodonitepink       #efbbcc

	pale rose-purple           palerosepurple          #b695c0

	pale rosolane purple       palerosolanepurple      #de6fa1

	pale russian blue          palerussianblue         #bcd4e6

	pale salmon color          palesalmoncolor         #fad6a5

	pale smoke gray            palesmokegray           #bfb8a5

	pale soft blue-violet      palesoftblueviolet      #8791bf

	pale soft blue-violet      palesoftblueviolet      #8c82b6

	pale sulphate green        palesulphategreen       #66ada4

	pale tiber green           paletibergreen          #c9dc89

	pale turquoise green       paleturquoisegreen      #96ded1

	pale turtle green          paleturtlegreen         #c9dc89

	pale turtle green          paleturtlegreen         #dadfb7

	pale varley's gray         palevarleysgray         #9690ab

	pale varley's gray         palevarleysgray         #aa98a9

	pale verbena violet        paleverbenaviolet       #dcd0ff

	pale veronese green        paleveronesegreen       #c9dc89

	pale veronese green        paleveronesegreen       #dadfb7

	pale vinaceous             palevinaceous           #dea5a4

	pale vinaceous-drab        palevinaceousdrab       #c4aead

	pale vinaceous-drab        palevinaceousdrab       #c1b6b3

	pale vinaceous-drab        palevinaceousdrab       #ad8884

	pale vinaceous-drab        palevinaceousdrab       #8f817f

	pale vinaceous-fawn        palevinaceousfawn       #c1b6b3

	pale vinaceous-fawn        palevinaceousfawn       #ecd5c5

	pale vinaceous-fawn        palevinaceousfawn       #c7ada3

	pale vinaceous-fawn        palevinaceousfawn       #c2ac99

	pale vinaceous-lilac       palevinaceouslilac      #efbbcc

	pale vinaceous-lilac       palevinaceouslilac      #e8ccd7

	pale vinaceous-pink        palevinaceouspink       #ecd5c5

	pale vinaceous-pink        palevinaceouspink       #c7ada3

	pale violet                paleviolet              #8791bf

	pale violet                paleviolet              #8c82b6

	pale violet-blue           palevioletblue          #4997d0

	pale violet-gray           palevioletgray          #8b8589

	pale violet-gray           palevioletgray          #848482

	pale violet-plumbeous      palevioletplumbeous     #91a3b0

	pale violet-plumbeous      palevioletplumbeous     #8c92ac

	pale viridine yellow       paleviridineyellow      #eae679

	pale viridine yellow       paleviridineyellow      #ebe8a4

	pale windsor blue          palewindsorblue         #70a3cc

	pale windsor blue          palewindsorblue         #8791bf

	pale wistaria blue         palewistariablue        #dcd0ff

	pale wistaria blue         palewistariablue        #8c82b6

	pale wistaria violet       palewistariaviolet      #8c82b6

	pale wistaria violet       palewistariaviolet      #9690ab

	pale yellow-green          paleyellowgreen         #c9dc89

	pale yellow-orange         paleyelloworange        #f4c2c2

	pallid blue-violet         pallidblueviolet        #b3bce2

	pallid blue-violet         pallidblueviolet        #8791bf

	pallid bluish violet       pallidbluishviolet      #b3bce2

	pallid bluish violet       pallidbluishviolet      #8791bf

	pallid bluish violet       pallidbluishviolet      #dcd0ff

	pallid bluish violet       pallidbluishviolet      #8c82b6

	pallid brownish drab       pallidbrownishdrab      #c1b6b3

	pallid grayish violet-blue pallidgrayishvioletblue #8791bf

	pallid grayish violet-blue pallidgrayishvioletblue #8c92ac

	pallid methyl blue         pallidmethylblue        #a1caf1

	pallid mouse gray          pallidmousegray         #c1b6b3

	pallid mouse gray          pallidmousegray         #b9b8b5

	pallid neutral gray        pallidneutralgray       #b9b8b5

	pallid purple-drab         pallidpurpledrab        #bfb9bd

	pallid purple-drab         pallidpurpledrab        #c3a6b1

	pallid purplish gray       pallidpurplishgray      #bfb9bd

	pallid quaker drab         pallidquakerdrab        #bfb9bd

	pallid soft blue-violet    pallidsoftblueviolet    #b3bce2

	pallid soft blue-violet    pallidsoftblueviolet    #8791bf

	pallid soft blue-violet    pallidsoftblueviolet    #dcd0ff

	pallid soft blue-violet    pallidsoftblueviolet    #8c82b6

	pallid vinaceous-drab      pallidvinaceousdrab     #bfb9bd

	pallid vinaceous-drab      pallidvinaceousdrab     #e8ccd7

	pallid violet              pallidviolet            #b3bce2

	pallid violet              pallidviolet            #8791bf

	pallid violet              pallidviolet            #c0c8e1

	pallid violet              pallidviolet            #8c92ac

	pallid violet              pallidviolet            #dcd0ff

	pallid violet              pallidviolet            #8c82b6

	pallid violet              pallidviolet            #c4c3dd

	pallid violet              pallidviolet            #9690ab

	pallid violet-blue         pallidvioletblue        #b3bce2

	pallid violet-blue         pallidvioletblue        #8791bf

	pansy purple               pansypurple             #673147

	pansy violet               pansyviolet             #875692

	paris blue                 parisblue               #0067a5

	paris blue                 parisblue               #00416a

	paris green                parisgreen              #93c592

	parrot green               parrotgreen             #8a9a5b

	parula blue                parulablue              #91a3b0

	parula blue                parulablue              #536878

	patent blue                patentblue              #004958

	payne's gray               paynesgray              #536878

	peach red                  peachred                #cb6d51

	peacock blue               peacockblue             #007791

	peacock green              peacockgreen            #8a9a5b

	pea green                  peagreen                #7d8984

	pearl blue                 pearlblue               #c0c8e1

	pearl blue                 pearlblue               #8c92ac

	pearl gray                 pearlgray               #b9b8b5

	pecan brown                pecanbrown              #a67b5b

	perilla purple             perillapurple           #673147

	persian blue               persianblue             #bcd4e6

	persian blue               persianblue             #91a3b0

	persian lilac              persianlilac            #c17e91

	petunia violet             petuniaviolet           #602f6b

	phenyl blue                phenylblue              #30267a

	phlox pink                 phloxpink               #b695c0

	phlox purple               phloxpurple             #9e4f88

	picric yellow              picricyellow            #e9e450

	pinard yellow              pinardyellow            #fada5e

	pinard yellow              pinardyellow            #f8de7e

	pinkish buff               pinkishbuff             #f4c2c2

	pinkish buff               pinkishbuff             #d9a6a9

	pinkish cinnamon           pinkishcinnamon         #e3a857

	pinkish cinnamon           pinkishcinnamon         #c19a6b

	pinkish vinaceous          pinkishvinaceous        #d9a6a9

	pistachio green            pistachiogreen          #679267

	pleroma violet             pleromaviolet           #7e73b8

	pleroma violet             pleromaviolet           #604e97

	plumbago blue              plumbagoblue            #9690ab

	plumbago blue              plumbagoblue            #aa98a9

	plumbago gray              plumbagogray            #8b8589

	plumbeous                  plumbeous               #91a3b0

	plumbeous                  plumbeous               #81878b

	plumbeous black            plumbeousblack          #36454f

	plumbeous black            plumbeousblack          #222222

	plum purple                plumpurple              #2f2140

	pois green                 poisgreen               #8f9779

	pompeian red               pompeianred             #ab4e52

	porcelain blue             porcelainblue           #66aabc

	porcelain green            porcelaingreen          #317873

	primrose yellow            primroseyellow          #eae679

	primuline yellow           primulineyellow         #d4af37

	prout's brown              proutsbrown             #6f4e37

	prune purple               prunepurple             #563c5c

	prussian blue              prussianblue            #0067a5

	prussian blue              prussianblue            #436b95

	prussian green             prussiangreen           #004b49

	prussian red               prussianred             #905d5d

	puritan gray               puritangray             #81878b

	purple                     purple                  #870074

	purple-drab                purpledrab              #796878

	purplish gray              purplishgray            #796878

	purplish gray              purplishgray            #5d555b

	purplish lilac             purplishlilac           #b784a7

	purplish lilac             purplishlilac           #d597ae

	purplish vinaceous         purplishvinaceous       #ad8884

	pyrite yellow              pyriteyellow            #98943e

	quaker drab                quakerdrab              #8b8589

	quaker drab                quakerdrab              #848482

	rainette green             rainettegreen           #8a9a5b

	raisin black               raisinblack             #50404d

	raisin purple              raisinpurple            #5d3954

	ramier blue                ramierblue              #9690ab

	ramier blue                ramierblue              #554c69

	ramier blue                ramierblue              #796878

	raw sienna                 rawsienna               #be8a3d

	raw sienna                 rawsienna               #996515

	reed yellow                reedyellow              #b9b459

	rejane green               rejanegreen             #93c592

	rejane green               rejanegreen             #679267

	rhodamine purple           rhodaminepurple         #ce4676

	rhodonite pink             rhodonitepink           #c17e91

	rinnemann's green          rinnemannsgreen         #8a9a5b

	rivage green               rivagegreen             #c9dc89

	rivage green               rivagegreen             #8a9a5b

	rocellin purple            rocellinpurple          #de6fa1

	rocellin purple            rocellinpurple          #c17e91

	roman green                romangreen              #665d1e

	rood's blue                roodsblue               #00a1c2

	rood's brown               roodsbrown              #977f73

	rood's brown               roodsbrown              #a67b5b

	rood's brown               roodsbrown              #6f4e37

	rood's lavender            roodslavender           #aa98a9

	rood's violet              roodsviolet             #702963

	rose color                 rosecolor               #de6fa1

	rose color                 rosecolor               #ce4676

	rose doree                 rosedoree               #e66761

	rose pink                  rosepink                #efbbcc

	rose-purple                rosepurple              #d399e6

	rose red                   rosered                 #be0032

	roslyn blue                roslynblue              #272458

	roslyn blue                roslynblue              #4e5180

	roslyn blue                roslynblue              #604e81

	rosolane pink              rosolanepink            #ffc8d6

	rosolane pink              rosolanepink            #efbbcc

	rosolane purple            rosolanepurple          #b3446c

	royal purple               royalpurple             #604e81

	rufous                     rufous                  #cb6d51

	russet                     russet                  #6f4e37

	russet-vinaceous           russetvinaceous         #905d5d

	russet-vinaceous           russetvinaceous         #a87c6d

	russian blue               russianblue             #bcd4e6

	russian blue               russianblue             #91a3b0

	russian green              russiangreen            #5e716a

	saccardo's olive           saccardosolive          #967117

	saccardo's olive           saccardosolive          #6c541e

	saccardo's slate           saccardosslate          #51585e

	saccardo's umber           saccardosumber          #7e6d5a

	saccardo's umber           saccardosumber          #6c541e

	saccardo's violet          saccardosviolet         #875692

	saccardo's violet          saccardosviolet         #86608e

	safrano pink               safranopink             #f88379

	sage green                 sagegreen               #8da399

	sailor blue                sailorblue              #545aa7

	salmon-buff                salmonbuff              #e3a857

	salmon-buff                salmonbuff              #fad6a5

	salmon color               salmoncolor             #d9a6a9

	salmon color               salmoncolor             #fab57f

	salmon color               salmoncolor             #d99058

	salmon-orange              salmonorange            #ed872d

	salvia blue                salviablue              #30267a

	sanford's brown            sanfordsbrown           #ae6938

	sanford's brown            sanfordsbrown           #80461b

	sayal brown                sayalbrown              #996515

	sayal brown                sayalbrown              #826644

	scarlet                    scarlet                 #d9603b

	scarlet-red                scarletred              #be0032

	scheele's green            scheelesgreen           #7e9f2e

	scheele's green            scheelesgreen           #8a9a5b

	schoenfeld's purple        schoenfeldspurple       #78184a

	sea-foam green             seafoamgreen            #ebe8a4

	sea-foam yellow            seafoamyellow           #eae679

	sea-foam yellow            seafoamyellow           #ebe8a4

	sea green                  seagreen                #317873

	seal brown                 sealbrown               #5c504f

	seal brown                 sealbrown               #674c47

	seashell pink              seashellpink            #ecd5c5

	seashell pink              seashellpink            #c2ac99

	seashell pink              seashellpink            #fad6a5

	sepia                      sepia                   #483c32

	sepia                      sepia                   #6c541e

	serpentine green           serpentinegreen         #867e36

	shamrock green             shamrockgreen           #679267

	shell pink                 shellpink               #ecd5c5

	shell pink                 shellpink               #c7ada3

	shrimp pink                shrimppink              #f4c2c2

	shrimp pink                shrimppink              #d9a6a9

	skobeloff green            skobeloffgreen          #007a74

	sky blue                   skyblue                 #70a3cc

	sky gray                   skygray                 #bcd4e6

	slate-black                slateblack              #222222

	slate-blue                 slateblue               #4c516d

	slate color                slatecolor              #555555

	slate-gray                 slategray               #555555

	slate-olive                slateolive              #7d8984

	slate-purple               slatepurple             #796878

	slate-purple               slatepurple             #836479

	slate-violet               slateviolet             #50404d

	slate-violet               slateviolet             #5d555b

	slate-violet               slateviolet             #796878

	smalt blue                 smaltblue               #30267a

	smoke gray                 smokegray               #c2b280

	smoke gray                 smokegray               #bfb8a5

	snuff brown                snuffbrown              #826644

	snuff brown                snuffbrown              #7e6d5a

	soft blue-violet           softblueviolet          #7e73b8

	soft bluish violet         softbluishviolet        #604e97

	sooty black                sootyblack              #555555

	sorghum brown              sorghumbrown            #674c47

	sorrento green             sorrentogreen           #317873

	spectrum blue              spectrumblue            #00a1c2

	spectrum red               spectrumred             #be0032

	spectrum violet            spectrumviolet          #604e97

	spinach green              spinachgreen            #8a9a5b

	spinel pink                spinelpink              #b3446c

	spinel red                 spinelred               #b3446c

	squill blue                squillblue              #70a3cc

	stone green                stonegreen              #679267

	storm gray                 stormgray               #7d8984

	strawberry pink            strawberrypink          #cb6d51

	straw yellow               strawyellow             #f8de7e

	strontian yellow           strontianyellow         #beb72e

	sudan brown                sudanbrown              #80461b

	sudan brown                sudanbrown              #a67b5b

	sudan brown                sudanbrown              #996515

	sulphate green             sulphategreen           #66ada4

	sulphate green             sulphategreen           #317873

	sulphine yellow            sulphineyellow          #ab9144

	sulphur yellow             sulphuryellow           #eae679

	sulphur yellow             sulphuryellow           #ebe8a4

	taupe brown                taupebrown              #50404d

	tawny                      tawny                   #ae6938

	tawny-olive                tawnyolive              #826644

	tawny-olive                tawnyolive              #967117

	tea green                  teagreen                #8f9779

	terra cotta                terracotta              #b4745e

	terre verte                terreverte              #5e716a

	testaceous                 testaceous              #b4745e

	testaceous                 testaceous              #a87c6d

	thulite pink               thulitepink             #e68fac

	tiber green                tibergreen              #93c592

	tilleul-buff               tilleulbuff             #ecd5c5

	tourmaline pink            tourmalinepink          #9e4f88

	tourmaline pink            tourmalinepink          #de6fa1

	turquoise green            turquoisegreen          #66ada4

	turtle green               turtlegreen             #8a9a5b

	tyrian blue                tyrianblue              #536878

	tyrian blue                tyrianblue              #4c516d

	tyrian pink                tyrianpink              #9e4f88

	tyrian rose                tyrianrose              #ce4676

	tyrolite green             tyrolitegreen           #66ada4

	ultramarine ash            ultramarineash          #0067a5

	ultramarine blue           ultramarineblue         #00a1c2

	urania blue                uraniablue              #2f2140

	urania blue                uraniablue              #554c69

	urania blue                uraniablue              #50404d

	vanderpoel's blue          vanderpoelsblue         #0067a5

	vanderpoel's blue          vanderpoelsblue         #436b95

	vanderpoel's green         vanderpoelsgreen        #bdda57

	vanderpoel's green         vanderpoelsgreen        #7e9f2e

	vanderpoel's violet        vanderpoelsviolet       #8c82b6

	vandyke brown              vandykebrown            #6f4e37

	vandyke brown              vandykebrown            #635147

	vandyke red                vandykered              #905d5d

	variscite green            variscitegreen          #8ed1b2

	varley's gray              varleysgray             #796878

	varley's green             varleysgreen            #355e3b

	venetian blue              venetianblue            #0067a5

	venetian pink              venetianpink            #f88379

	venice green               venicegreen             #00a693

	verbena violet             verbenaviolet           #9690ab

	verdigris green            verdigrisgreen          #6aab8e

	vernonia purple            vernoniapurple          #915f6d

	verona brown               veronabrown             #6f4e37

	veronese green             veronesegreen           #c9dc89

	veronese green             veronesegreen           #8a9a5b

	vetiver green              vetivergreen            #8a9a5b

	vetiver green              vetivergreen            #8f9779

	victoria lake              victorialake            #722f37

	victoria lake              victorialake            #3f1728

	vinaceous                  vinaceous               #ea9399

	vinaceous                  vinaceous               #e4717a

	vinaceous                  vinaceous               #dea5a4

	vinaceous                  vinaceous               #c08081

	vinaceous-brown            vinaceousbrown          #905d5d

	vinaceous-buff             vinaceousbuff           #f4c2c2

	vinaceous-buff             vinaceousbuff           #d9a6a9

	vinaceous-buff             vinaceousbuff           #ecd5c5

	vinaceous-buff             vinaceousbuff           #c2ac99

	vinaceous-cinnamon         vinaceouscinnamon       #d9a6a9

	vinaceous-cinnamon         vinaceouscinnamon       #c19a6b

	vinaceous-drab             vinaceousdrab           #905d5d

	vinaceous-fawn             vinaceousfawn           #d9a6a9

	vinaceous-fawn             vinaceousfawn           #c7ada3

	vinaceous-gray             vinaceousgray           #aa98a9

	vinaceous-gray             vinaceousgray           #796878

	vinaceous-lavender         vinaceouslavender       #dea5a4

	vinaceous-lilac            vinaceouslilac          #c17e91

	vinaceous-lilac            vinaceouslilac          #915f6d

	vinaceous-pink             vinaceouspink           #b4745e

	vinaceous-pink             vinaceouspink           #a87c6d

	vinaceous-purple           vinaceouspurple         #a8516e

	vinaceous-purple           vinaceouspurple         #915f6d

	vinaceous-purple           vinaceouspurple         #836479

	vinaceous-rufous           vinaceousrufous         #9e4732

	vinaceous-russet           vinaceousrusset         #a87c6d

	vinaceous-russet           vinaceousrusset         #79443b

	vinaceous-slate            vinaceousslate          #796878

	vinaceous-tawny            vinaceoustawny          #b4745e

	violet carmine             violetcarmine           #673147

	violet-gray                violetgray              #8b8589

	violet-plumbeous           violetplumbeous         #8c92ac

	violet-purple              violetpurple            #9a4eae

	violet-slate               violetslate             #536878

	violet ultramarine         violetultramarine       #545aa7

	violet ultramarine         violetultramarine       #604e97

	viridian green             viridiangreen           #3eb489

	viridian green             viridiangreen           #007959

	viridine green             viridinegreen           #bdda57

	viridine yellow            viridineyellow          #beb72e

	viridine yellow            viridineyellow          #b9b459

	vivid green                vividgreen              #3eb489

	wall green                 wallgreen               #317873

	walnut brown               walnutbrown             #79443b

	walnut brown               walnutbrown             #674c47

	warbler green              warblergreen            #867e36

	warm blackish brown        warmblackishbrown       #543d3f

	warm blackish brown        warmblackishbrown       #5c504f

	warm buff                  warmbuff                #e3a857

	warm buff                  warmbuff                #fad6a5

	warm buff                  warmbuff                #c9ae5d

	warm sepia                 warmsepia               #635147

	water green                watergreen              #dadfb7

	wax yellow                 waxyellow               #d4af37

	wedgewood blue             wedgewoodblue           #a1caf1

	wedgewood blue             wedgewoodblue           #70a3cc

	wedgewood blue             wedgewoodblue           #b3bce2

	wedgewood blue             wedgewoodblue           #8791bf

	white                      white                   #f2f3f4

	windsor blue               windsorblue             #4e5180

	winter green               wintergreen             #679267

	wistaria blue              wistariablue            #6c79b8

	wistaria blue              wistariablue            #7e73b8

	wistaria violet            wistariaviolet          #7e73b8

	wood brown                 woodbrown               #c19a6b

	wood brown                 woodbrown               #ae9b82

	xanthine orange            xanthineorange          #be6516

	yale blue                  yaleblue                #70a3cc

	yellow green               yellowgreen             #bdda57

	yellowish citrine          yellowishcitrine        #98943e

	yellowish glaucous         yellowishglaucous       #dadfb7

	yellowish oil green        yellowishoilgreen       #867e36

	yellowish olive            yellowisholive          #867e36

	yellowish olive            yellowisholive          #665d1e

	yellow ochre               yellowochre             #be8a3d

	yew green                  yewgreen                #4a5d23

	yvette violet              yvetteviolet            #554c69

	zinc green                 zincgreen               #3b7861

	zinc orange                zincorange              #ed872d

	zinc orange                zincorange              #d99058


=cut

sub _load_color_list() {
    return [
['nbs-iscc-r:absinthegreen.120','absinthegreen','absinthe green',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-r:acajoured.19','acajoured','acajou red',[144,93,93],'905d5d',9461085],
['nbs-iscc-r:acetinblue.173-','acetinblue','acetin blue ',[54,117,136],'367588',3569032],
['nbs-iscc-r:acetinblue.204','acetinblue','acetin blue ',[76,81,109],'4c516d',5001581],
['nbs-iscc-r:ackermannsgreen.145','ackermannsgreen','ackermann\'s green',[59,120,97],'3b7861',3897441],
['nbs-iscc-r:aconiteviolet.223','aconiteviolet','aconite violet',[134,96,142],'86608e',8806542],
['nbs-iscc-r:ageratumviolet.223','ageratumviolet','ageratum violet',[134,96,142],'86608e',8806542],
['nbs-iscc-r:aliceblue.185','aliceblue','alice blue',[145,163,176],'91a3b0',9544624],
['nbs-iscc-r:alizarineblue.182','alizarineblue','alizarine blue',[67,107,149],'436b95',4418453],
['nbs-iscc-r:alizarinepink.2','alizarinepink','alizarine pink',[234,147,153],'ea9399',15373209],
['nbs-iscc-r:alizarinepink.3','alizarinepink','alizarine pink',[228,113,122],'e4717a',14971258],
['nbs-iscc-r:amaranthpink.248','amaranthpink','amaranth pink',[222,111,161],'de6fa1',14577569],
['nbs-iscc-r:amaranthpurple.256','amaranthpurple','amaranth purple',[120,24,74],'78184a',7870538],
['nbs-iscc-r:amberbrown.55','amberbrown','amber brown',[128,70,27],'80461b',8406555],
['nbs-iscc-r:amberyellow.83','amberyellow','amber yellow ',[250,218,94],'fada5e',16439902],
['nbs-iscc-r:amberyellow.84','amberyellow','amber yellow ',[212,175,55],'d4af37',13938487],
['nbs-iscc-r:amberyellow.86','amberyellow','amber yellow ',[248,222,126],'f8de7e',16309886],
['nbs-iscc-r:amberyellow.87','amberyellow','amber yellow ',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-r:americangreen.136','americangreen','american green',[103,146,103],'679267',6787687],
['nbs-iscc-r:amethystviolet.216','amethystviolet','amethyst violet',[154,78,174],'9a4eae',10112686],
['nbs-iscc-r:amparoblue.195','amparoblue','amparo blue',[108,121,184],'6c79b8',7109048],
['nbs-iscc-r:amparopurple.217','amparopurple','amparo purple',[211,153,230],'d399e6',13867494],
['nbs-iscc-r:amparopurple.218','amparopurple','amparo purple',[135,86,146],'875692',8869522],
['nbs-iscc-r:andovergreen.122','andovergreen','andover green',[143,151,121],'8f9779',9410425],
['nbs-iscc-r:anilineblack.229','anilineblack','aniline black',[80,64,77],'50404d',5259341],
['nbs-iscc-r:anilineblack.234','anilineblack','aniline black',[93,85,91],'5d555b',6116699],
['nbs-iscc-r:anilinelilac.210','anilinelilac','aniline lilac',[140,130,182],'8c82b6',9208502],
['nbs-iscc-r:anilineyellow.85','anilineyellow','aniline yellow',[175,141,19],'af8d13',11504915],
['nbs-iscc-r:anthracenegreen.164','anthracenegreen','anthracene green',[49,120,115],'317873',3242099],
['nbs-iscc-r:anthracenegreen.165','anthracenegreen','anthracene green',[0,75,73],'004b49',19273],
['nbs-iscc-r:anthracenepurple.229','anthracenepurple','anthracene purple',[80,64,77],'50404d',5259341],
['nbs-iscc-r:anthracenepurple.259','anthracenepurple','anthracene purple',[103,49,71],'673147',6762823],
['nbs-iscc-r:anthraceneviolet.211','anthraceneviolet','anthracene violet',[96,78,129],'604e81',6311553],
['nbs-iscc-r:antimonyyellow.71','antimonyyellow','antimony yellow',[227,168,87],'e3a857',14919767],
['nbs-iscc-r:antiquebrown.55','antiquebrown','antique brown',[128,70,27],'80461b',8406555],
['nbs-iscc-r:antiquebrown.58','antiquebrown','antique brown',[111,78,55],'6f4e37',7294519],
['nbs-iscc-r:antiquegreen.150','antiquegreen','antique green',[94,113,106],'5e716a',6189418],
['nbs-iscc-r:antiquegreen.151','antiquegreen','antique green',[58,75,71],'3a4b47',3820359],
['nbs-iscc-r:antwerpblue.182','antwerpblue','antwerp blue',[67,107,149],'436b95',4418453],
['nbs-iscc-r:antwerpblue.183','antwerpblue','antwerp blue',[0,48,78],'00304e',12366],
['nbs-iscc-r:antwerpblue.186','antwerpblue','antwerp blue',[83,104,120],'536878',5466232],
['nbs-iscc-r:applegreen.117','applegreen','apple green',[126,159,46],'7e9f2e',8298286],
['nbs-iscc-r:apricotbuff.53','apricotbuff','apricot buff',[217,144,88],'d99058',14258264],
['nbs-iscc-r:apricotorange.53','apricotorange','apricot orange',[217,144,88],'d99058',14258264],
['nbs-iscc-r:apricotyellow.84','apricotyellow','apricot yellow',[212,175,55],'d4af37',13938487],
['nbs-iscc-r:argusbrown.58','argusbrown','argus brown',[111,78,55],'6f4e37',7294519],
['nbs-iscc-r:argylepurple.241','argylepurple','argyle purple',[145,92,131],'915c83',9526403],
['nbs-iscc-r:armybrown.58','armybrown','army brown',[111,78,55],'6f4e37',7294519],
['nbs-iscc-r:armybrown.61','armybrown','army brown',[99,81,71],'635147',6508871],
['nbs-iscc-r:artemesiagreen.155','artemesiagreen','artemesia green',[125,137,132],'7d8984',8227204],
['nbs-iscc-r:asphodelgreen.120','asphodelgreen','asphodel green',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-r:asterpurple.256','asterpurple','aster purple',[120,24,74],'78184a',7870538],
['nbs-iscc-r:auburn.43','auburn','auburn',[121,68,59],'79443b',7947323],
['nbs-iscc-r:auriculapurple.259','auriculapurple','auricula purple',[103,49,71],'673147',6762823],
['nbs-iscc-r:avellaneous.76','avellaneous','avellaneous',[193,154,107],'c19a6b',12687979],
['nbs-iscc-r:avellaneous.79','avellaneous','avellaneous',[174,155,130],'ae9b82',11443074],
['nbs-iscc-r:azuriteblue.200','azuriteblue','azurite blue',[78,81,128],'4e5180',5132672],
['nbs-iscc-r:bariumyellow.101','bariumyellow','barium yellow',[234,230,121],'eae679',15394425],
['nbs-iscc-r:bariumyellow.102','bariumyellow','barium yellow',[185,180,89],'b9b459',12170329],
['nbs-iscc-r:barytayellow.86','barytayellow','baryta yellow',[248,222,126],'f8de7e',16309886],
['nbs-iscc-r:bay.43','bay','bay',[121,68,59],'79443b',7947323],
['nbs-iscc-r:begoniarose.27','begoniarose','begonia rose',[230,103,97],'e66761',15099745],
['nbs-iscc-r:benzobrown.45','benzobrown','benzo brown',[151,127,115],'977f73',9928563],
['nbs-iscc-r:benzobrown.60','benzobrown','benzo brown',[149,128,112],'958070',9797744],
['nbs-iscc-r:benzobrown.61','benzobrown','benzo brown',[99,81,71],'635147',6508871],
['nbs-iscc-r:benzolgreen.160','benzolgreen','benzol green',[0,122,116],'007a74',31348],
['nbs-iscc-r:berlinblue.183','berlinblue','berlin blue',[0,48,78],'00304e',12366],
['nbs-iscc-r:berylblue.171','berylblue','beryl blue',[156,209,220],'9cd1dc',10277340],
['nbs-iscc-r:berylgreen.159','berylgreen','beryl green',[0,166,147],'00a693',42643],
['nbs-iscc-r:bicegreen.120','bicegreen','bice green',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-r:biscaygreen.120','biscaygreen','biscay green',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-r:bishopspurple.241','bishopspurple','bishop\'s purple',[145,92,131],'915c83',9526403],
['nbs-iscc-r:bister.80','bister','bister',[126,109,90],'7e6d5a',8285530],
['nbs-iscc-r:bister.81','bister','bister',[72,60,50],'483c32',4734002],
['nbs-iscc-r:bittersweetorange.37','bittersweetorange','bittersweet orange',[203,109,81],'cb6d51',13331793],
['nbs-iscc-r:bittersweetpink.53','bittersweetpink','bittersweet pink',[217,144,88],'d99058',14258264],
['nbs-iscc-r:black.267','black','black',[34,34,34],'222222',2236962],
['nbs-iscc-r:blackishgreenblue.174','blackishgreenblue','blackish green-blue',[0,73,88],'004958',18776],
['nbs-iscc-r:blackishgreengray.267','blackishgreengray','blackish green-gray',[34,34,34],'222222',2236962],
['nbs-iscc-r:blackishmousegray.64','blackishmousegray','blackish mouse-gray',[91,80,79],'5b504f',5984335],
['nbs-iscc-r:blackishmousegray.266','blackishmousegray','blackish mouse-gray',[85,85,85],'555555',5592405],
['nbs-iscc-r:blackishplumbeous.192','blackishplumbeous','blackish plumbeous',[81,88,94],'51585e',5331038],
['nbs-iscc-r:blackishpurple.242','blackishpurple','blackish purple',[93,57,84],'5d3954',6109524],
['nbs-iscc-r:blackishpurple.243','blackishpurple','blackish purple',[52,23,49],'341731',3413809],
['nbs-iscc-r:blackishredpurple.259','blackishredpurple','blackish red-purple',[103,49,71],'673147',6762823],
['nbs-iscc-r:blackishslate.266','blackishslate','blackish slate',[85,85,85],'555555',5592405],
['nbs-iscc-r:blackishviolet.211','blackishviolet','blackish violet',[96,78,129],'604e81',6311553],
['nbs-iscc-r:blackishviolet.212','blackishviolet','blackish violet',[47,33,64],'2f2140',3088704],
['nbs-iscc-r:blackishviolet.215','blackishviolet','blackish violet',[85,76,105],'554c69',5590121],
['nbs-iscc-r:blackishvioletgray.234','blackishvioletgray','blackish violet-gray',[93,85,91],'5d555b',6116699],
['nbs-iscc-r:blackishvioletgray.266','blackishvioletgray','blackish violet-gray',[85,85,85],'555555',5592405],
['nbs-iscc-r:blancsblue.182','blancsblue','blanc\'s blue',[67,107,149],'436b95',4418453],
['nbs-iscc-r:blancsviolet.215','blancsviolet','blanc\'s violet',[85,76,105],'554c69',5590121],
['nbs-iscc-r:blancsviolet.228','blancsviolet','blanc\'s violet',[121,104,120],'796878',7956600],
['nbs-iscc-r:blancsviolet.229','blancsviolet','blanc\'s violet',[80,64,77],'50404d',5259341],
['nbs-iscc-r:blueviolet.194','blueviolet','blue-violet',[48,38,122],'30267a',3155578],
['nbs-iscc-r:blueviolet.205','blueviolet','blue-violet',[144,101,202],'9065ca',9463242],
['nbs-iscc-r:bluevioletblack.234','bluevioletblack','blue-violet black',[93,85,91],'5d555b',6116699],
['nbs-iscc-r:bluevioletblack.266','bluevioletblack','blue-violet black',[85,85,85],'555555',5592405],
['nbs-iscc-r:bluishblack.187','bluishblack','bluish black',[54,69,79],'36454f',3556687],
['nbs-iscc-r:bluishblack.188','bluishblack','bluish black',[32,40,48],'202830',2107440],
['nbs-iscc-r:bluishglaucous.148','bluishglaucous','bluish glaucous',[199,230,215],'c7e6d7',13100759],
['nbs-iscc-r:bluishglaucous.162','bluishglaucous','bluish glaucous',[150,222,209],'96ded1',9887441],
['nbs-iscc-r:bluishgraygreen.149','bluishgraygreen','bluish gray-green',[141,163,153],'8da399',9282457],
['nbs-iscc-r:bluishgraygreen.150','bluishgraygreen','bluish gray-green',[94,113,106],'5e716a',6189418],
['nbs-iscc-r:bluishlavender.210','bluishlavender','bluish lavender',[140,130,182],'8c82b6',9208502],
['nbs-iscc-r:bluishslateblack.183','bluishslateblack','bluish slate-black',[0,48,78],'00304e',12366],
['nbs-iscc-r:bluishslateblack.187','bluishslateblack','bluish slate-black',[54,69,79],'36454f',3556687],
['nbs-iscc-r:bluishviolet.194','bluishviolet','bluish violet',[48,38,122],'30267a',3155578],
['nbs-iscc-r:bluishviolet.205','bluishviolet','bluish violet',[144,101,202],'9065ca',9463242],
['nbs-iscc-r:bonebrown.61','bonebrown','bone brown',[99,81,71],'635147',6508871],
['nbs-iscc-r:bonebrown.64','bonebrown','bone brown',[91,80,79],'5b504f',5984335],
['nbs-iscc-r:bordeaux.259','bordeaux','bordeaux',[103,49,71],'673147',6762823],
['nbs-iscc-r:bottlegreen.146','bottlegreen','bottle green',[27,77,62],'1b4d3e',1789246],
['nbs-iscc-r:bradleysblue.176','bradleysblue','bradley\'s blue',[0,161,194],'00a1c2',41410],
['nbs-iscc-r:bradleysviolet.206','bradleysviolet','bradley\'s violet',[126,115,184],'7e73b8',8287160],
['nbs-iscc-r:bradleysviolet.207','bradleysviolet','bradley\'s violet',[96,78,151],'604e97',6311575],
['nbs-iscc-r:bradleysviolet.210','bradleysviolet','bradley\'s violet',[140,130,182],'8c82b6',9208502],
['nbs-iscc-r:bradleysviolet.211','bradleysviolet','bradley\'s violet',[96,78,129],'604e81',6311553],
['nbs-iscc-r:brazilred.38','brazilred','brazil red',[158,71,50],'9e4732',10372914],
['nbs-iscc-r:bremenblue.172','bremenblue','bremen blue',[102,170,188],'66aabc',6728380],
['nbs-iscc-r:brickred.43','brickred','brick red',[121,68,59],'79443b',7947323],
['nbs-iscc-r:brightchalcedonyyellow.116','brightchalcedonyyellow','bright chalcedony yellow',[189,218,87],'bdda57',12442199],
['nbs-iscc-r:brightchalcedonyyellow.119','brightchalcedonyyellow','bright chalcedony yellow',[201,220,137],'c9dc89',13229193],
['nbs-iscc-r:brightgreenyellow.115','brightgreenyellow','bright green-yellow',[141,182,0],'8db600',9287168],
['nbs-iscc-r:brownishdrab.45','brownishdrab','brownish drab',[151,127,115],'977f73',9928563],
['nbs-iscc-r:brownishdrab.70','brownishdrab','brownish drab',[149,128,112],'958070',9797744],
['nbs-iscc-r:brownisholive.95','brownisholive','brownish olive',[108,84,30],'6c541e',7099422],
['nbs-iscc-r:brownishvinaceous.18','brownishvinaceous','brownish vinaceous',[173,136,132],'ad8884',11372676],
['nbs-iscc-r:brusselsbrown.58','brusselsbrown','brussels brown',[111,78,55],'6f4e37',7294519],
['nbs-iscc-r:buckthornbrown.74','buckthornbrown','buckthorn brown',[153,101,21],'996515',10052885],
['nbs-iscc-r:buffpinkorbuffpink.29','buffpinkorbuffpink','buff-pink or buff pink',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-r:buffybrown.76','buffybrown','buffy brown ',[193,154,107],'c19a6b',12687979],
['nbs-iscc-r:buffybrown.77','buffybrown','buffy brown ',[130,102,68],'826644',8545860],
['nbs-iscc-r:buffycitrine.94','buffycitrine','buffy citrine',[150,113,23],'967117',9859351],
['nbs-iscc-r:buffyelloworbuffyellow.86','buffyelloworbuffyellow','buff-yellow or buff yellow',[248,222,126],'f8de7e',16309886],
['nbs-iscc-r:buffyelloworbuffyellow.87','buffyelloworbuffyellow','buff-yellow or buff yellow',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-r:buffyolive.106','buffyolive','buffy olive',[134,126,54],'867e36',8814134],
['nbs-iscc-r:burnblue.180','burnblue','burn blue',[161,202,241],'a1caf1',10603249],
['nbs-iscc-r:burnblue.181','burnblue','burn blue',[112,163,204],'70a3cc',7381964],
['nbs-iscc-r:burnblue.198','burnblue','burn blue',[179,188,226],'b3bce2',11779298],
['nbs-iscc-r:burnblue.199','burnblue','burn blue',[135,145,191],'8791bf',8884671],
['nbs-iscc-r:burntlake.242','burntlake','burnt lake',[93,57,84],'5d3954',6109524],
['nbs-iscc-r:burntsienna.40','burntsienna','burnt sienna',[136,45,23],'882d17',8924439],
['nbs-iscc-r:burntumber.46','burntumber','burnt umber',[103,76,71],'674c47',6769735],
['nbs-iscc-r:cacaobrown.42','cacaobrown','cacao brown',[168,124,109],'a87c6d',11041901],
['nbs-iscc-r:cadetblue.178','cadetblue','cadet blue',[0,103,165],'0067a5',26533],
['nbs-iscc-r:cadetblue.182','cadetblue','cadet blue',[67,107,149],'436b95',4418453],
['nbs-iscc-r:cadetgray.185','cadetgray','cadet gray',[145,163,176],'91a3b0',9544624],
['nbs-iscc-r:cadmiumorange.48','cadmiumorange','cadmium orange',[243,132,0],'f38400',15959040],
['nbs-iscc-r:cadmiumorange.50','cadmiumorange','cadmium orange',[237,135,45],'ed872d',15566637],
['nbs-iscc-r:cadmiumyellow.50','cadmiumyellow','cadmium yellow',[237,135,45],'ed872d',15566637],
['nbs-iscc-r:cadmiumyellow.68','cadmiumyellow','cadmium yellow',[234,162,33],'eaa221',15376929],
['nbs-iscc-r:calamineblue.171','calamineblue','calamine blue',[156,209,220],'9cd1dc',10277340],
['nbs-iscc-r:calamineblue.172','calamineblue','calamine blue',[102,170,188],'66aabc',6728380],
['nbs-iscc-r:callagreen.107','callagreen','calla green',[102,93,30],'665d1e',6708510],
['nbs-iscc-r:callistegreen.117','callistegreen','calliste green',[126,159,46],'7e9f2e',8298286],
['nbs-iscc-r:cameobrown.43','cameobrown','cameo brown',[121,68,59],'79443b',7947323],
['nbs-iscc-r:cameobrown.46','cameobrown','cameo brown',[103,76,71],'674c47',6769735],
['nbs-iscc-r:cameopink.246','cameopink','cameo pink',[255,200,214],'ffc8d6',16763094],
['nbs-iscc-r:cameopink.249','cameopink','cameo pink',[239,187,204],'efbbcc',15711180],
['nbs-iscc-r:capriblue.173','capriblue','capri blue ',[54,117,136],'367588',3569032],
['nbs-iscc-r:capucinebuff.70','capucinebuff','capucine buff ',[251,201,127],'fbc97f',16501119],
['nbs-iscc-r:capucinebuff.71','capucinebuff','capucine buff ',[227,168,87],'e3a857',14919767],
['nbs-iscc-r:capucinebuff.73','capucinebuff','capucine buff ',[250,214,165],'fad6a5',16438949],
['nbs-iscc-r:capucineorange.53','capucineorange','capucine orange ',[217,144,88],'d99058',14258264],
['nbs-iscc-r:capucineyellow.68','capucineyellow','capucine yellow ',[234,162,33],'eaa221',15376929],
['nbs-iscc-r:carmine.11','carmine','carmine',[190,0,50],'be0032',12451890],
['nbs-iscc-r:carnelianred.35','carnelianred','carnelian red',[217,96,59],'d9603b',14245947],
['nbs-iscc-r:carobbrown.44','carobbrown','carob brown',[62,29,30],'3e1d1e',4070686],
['nbs-iscc-r:carrotred.37','carrotred','carrot red',[203,109,81],'cb6d51',13331793],
['nbs-iscc-r:cartridgebuff.89','cartridgebuff','cartridge buff',[243,229,171],'f3e5ab',15984043],
['nbs-iscc-r:castorgray.155','castorgray','castor gray',[125,137,132],'7d8984',8227204],
['nbs-iscc-r:castorgray.265','castorgray','castor gray',[132,132,130],'848482',8684674],
['nbs-iscc-r:cedargreen.120','cedargreen','cedar green',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-r:cedargreen.125','cedargreen','cedar green',[74,93,35],'4a5d23',4873507],
['nbs-iscc-r:celandinegreen.149','celandinegreen','celandine green',[141,163,153],'8da399',9282457],
['nbs-iscc-r:cendreblue.168','cendreblue','cendre blue',[35,158,186],'239eba',2334394],
['nbs-iscc-r:cendregreen.130','cendregreen','cendre green',[131,211,125],'83d37d',8639357],
['nbs-iscc-r:ceruleanblue.178','ceruleanblue','cerulean blue',[0,103,165],'0067a5',26533],
['nbs-iscc-r:chaeturablack.81','chaeturablack','chaetura black',[72,60,50],'483c32',4734002],
['nbs-iscc-r:chaeturadrab.64','chaeturadrab','chaetura drab',[91,80,79],'5b504f',5984335],
['nbs-iscc-r:chalcedonyyellow.101','chalcedonyyellow','chalcedony yellow',[234,230,121],'eae679',15394425],
['nbs-iscc-r:chamois.87','chamois','chamois',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-r:chapmansblue.182','chapmansblue','chapman\'s blue',[67,107,149],'436b95',4418453],
['nbs-iscc-r:chartreuseyellow.101','chartreuseyellow','chartreuse yellow',[234,230,121],'eae679',15394425],
['nbs-iscc-r:chatenaypink.5','chatenaypink','chatenay pink',[222,165,164],'dea5a4',14591396],
['nbs-iscc-r:chatenaypink.29','chatenaypink','chatenay pink',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-r:chessyliteblue.173','chessyliteblue','chessylite blue',[54,117,136],'367588',3569032],
['nbs-iscc-r:chessyliteblue.174','chessyliteblue','chessylite blue',[0,73,88],'004958',18776],
['nbs-iscc-r:chessyliteblue.182','chessyliteblue','chessylite blue',[67,107,149],'436b95',4418453],
['nbs-iscc-r:chessyliteblue.186','chessyliteblue','chessylite blue',[83,104,120],'536878',5466232],
['nbs-iscc-r:chestnut.43','chestnut','chestnut',[121,68,59],'79443b',7947323],
['nbs-iscc-r:chestnutbrown.43','chestnutbrown','chestnut-brown',[121,68,59],'79443b',7947323],
['nbs-iscc-r:chestnutbrown.46','chestnutbrown','chestnut-brown',[103,76,71],'674c47',6769735],
['nbs-iscc-r:chicoryblue.199','chicoryblue','chicory blue',[135,145,191],'8791bf',8884671],
['nbs-iscc-r:chicoryblue.210','chicoryblue','chicory blue',[140,130,182],'8c82b6',9208502],
['nbs-iscc-r:chinablue.182','chinablue','china blue',[67,107,149],'436b95',4418453],
['nbs-iscc-r:chineseviolet.223','chineseviolet','chinese violet',[134,96,142],'86608e',8806542],
['nbs-iscc-r:chocolate.46','chocolate','chocolate',[103,76,71],'674c47',6769735],
['nbs-iscc-r:chromiumgreen.120','chromiumgreen','chromium green',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-r:chrysolitegreen.120','chrysolitegreen','chrysolite green',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-r:chrysopraisegreen.135','chrysopraisegreen','chrysopraise green',[147,197,146],'93c592',9684370],
['nbs-iscc-r:cinereous.190','cinereous','cinereous',[180,188,192],'b4bcc0',11844800],
['nbs-iscc-r:cinereous.191','cinereous','cinereous',[129,135,139],'81878b',8488843],
['nbs-iscc-r:cinnamon.57','cinnamon','cinnamon',[166,123,91],'a67b5b',10910555],
['nbs-iscc-r:cinnamon.72','cinnamon','cinnamon',[190,138,61],'be8a3d',12487229],
['nbs-iscc-r:cinnamonbrown.58','cinnamonbrown','cinnamon-brown',[111,78,55],'6f4e37',7294519],
['nbs-iscc-r:cinnamonbuff.71','cinnamonbuff','cinnamon-buff',[227,168,87],'e3a857',14919767],
['nbs-iscc-r:cinnamonbuff.76','cinnamonbuff','cinnamon-buff',[193,154,107],'c19a6b',12687979],
['nbs-iscc-r:cinnamonbuff.87','cinnamonbuff','cinnamon-buff',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-r:cinnamondrab.45','cinnamondrab','cinnamon-drab',[151,127,115],'977f73',9928563],
['nbs-iscc-r:cinnamondrab.60','cinnamondrab','cinnamon-drab',[149,128,112],'958070',9797744],
['nbs-iscc-r:cinnamonrufous.54','cinnamonrufous','cinnamon-rufous',[174,105,56],'ae6938',11430200],
['nbs-iscc-r:citrine.106','citrine','citrine',[134,126,54],'867e36',8814134],
['nbs-iscc-r:citrine.107','citrine','citrine',[102,93,30],'665d1e',6708510],
['nbs-iscc-r:citrinedrab.106','citrinedrab','citrine-drab',[134,126,54],'867e36',8814134],
['nbs-iscc-r:citrongreen.102','citrongreen','citron green',[185,180,89],'b9b459',12170329],
['nbs-iscc-r:citrongreen.105','citrongreen','citron green',[185,181,125],'b9b57d',12170621],
['nbs-iscc-r:citronyellow.98','citronyellow','citron yellow',[233,228,80],'e9e450',15328336],
['nbs-iscc-r:citronyellow.99','citronyellow','citron yellow',[190,183,46],'beb72e',12498734],
['nbs-iscc-r:citronyellow.101','citronyellow','citron yellow',[234,230,121],'eae679',15394425],
['nbs-iscc-r:citronyellow.102','citronyellow','citron yellow',[185,180,89],'b9b459',12170329],
['nbs-iscc-r:civettegreen.136','civettegreen','civette green',[103,146,103],'679267',6787687],
['nbs-iscc-r:claretbrown.43','claretbrown','claret brown',[121,68,59],'79443b',7947323],
['nbs-iscc-r:claycolor.74','claycolor','clay color',[153,101,21],'996515',10052885],
['nbs-iscc-r:claycolor.77','claycolor','clay color',[130,102,68],'826644',8545860],
['nbs-iscc-r:claycolor.94','claycolor','clay color',[150,113,23],'967117',9859351],
['nbs-iscc-r:clearcadetblue.177','clearcadetblue','clear cadet blue',[73,151,208],'4997d0',4822992],
['nbs-iscc-r:clearcadetblue.178','clearcadetblue','clear cadet blue',[0,103,165],'0067a5',26533],
['nbs-iscc-r:cleardullgreenyellow.116','cleardullgreenyellow','clear dull green-yellow',[189,218,87],'bdda57',12442199],
['nbs-iscc-r:cleardullgreenyellow.117','cleardullgreenyellow','clear dull green-yellow',[126,159,46],'7e9f2e',8298286],
['nbs-iscc-r:cleardullgreenyellow.119','cleardullgreenyellow','clear dull green-yellow',[201,220,137],'c9dc89',13229193],
['nbs-iscc-r:cleardullgreenyellow.120','cleardullgreenyellow','clear dull green-yellow',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-r:clearfluoritegreen.135','clearfluoritegreen','clear fluorite green',[147,197,146],'93c592',9684370],
['nbs-iscc-r:cleargreenbluegray.185','cleargreenbluegray','clear green-blue gray',[145,163,176],'91a3b0',9544624],
['nbs-iscc-r:cleargreenbluegray.191','cleargreenbluegray','clear green-blue gray',[129,135,139],'81878b',8488843],
['nbs-iscc-r:clearpaynesgray.186','clearpaynesgray','clear payne\'s gray ',[83,104,120],'536878',5466232],
['nbs-iscc-r:clearpaynesgray.191','clearpaynesgray','clear payne\'s gray ',[129,135,139],'81878b',8488843],
['nbs-iscc-r:clearwindsorblue.199','clearwindsorblue','clear windsor blue',[135,145,191],'8791bf',8884671],
['nbs-iscc-r:clearyellowgreen.116','clearyellowgreen','clear yellow-green',[189,218,87],'bdda57',12442199],
['nbs-iscc-r:clovebrown.81','clovebrown','clove brown',[72,60,50],'483c32',4734002],
['nbs-iscc-r:cobaltgreen.144','cobaltgreen','cobalt green',[106,171,142],'6aab8e',6990734],
['nbs-iscc-r:colonialbuff.86','colonialbuff','colonial buff',[248,222,126],'f8de7e',16309886],
['nbs-iscc-r:columbiablue.181','columbiablue','columbia blue',[112,163,204],'70a3cc',7381964],
['nbs-iscc-r:columbiablue.182','columbiablue','columbia blue',[67,107,149],'436b95',4418453],
['nbs-iscc-r:commelinablue.194','commelinablue','commelina blue',[48,38,122],'30267a',3155578],
['nbs-iscc-r:congopink.26','congopink','congo pink',[248,131,121],'f88379',16286585],
['nbs-iscc-r:congopink.29','congopink','congo pink',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-r:coralpink.26','coralpink','coral pink',[248,131,121],'f88379',16286585],
['nbs-iscc-r:coralred.37','coralred','coral red',[203,109,81],'cb6d51',13331793],
['nbs-iscc-r:corinthianpink.5','corinthianpink','corinthian pink',[222,165,164],'dea5a4',14591396],
['nbs-iscc-r:corinthianpurple.259','corinthianpurple','corinthian purple',[103,49,71],'673147',6762823],
['nbs-iscc-r:corinthianred.19','corinthianred','corinthian red',[144,93,93],'905d5d',9461085],
['nbs-iscc-r:cornflowerblue.194','cornflowerblue','cornflower blue',[48,38,122],'30267a',3155578],
['nbs-iscc-r:corydalisgreen.119','corydalisgreen','corydalis green',[201,220,137],'c9dc89',13229193],
['nbs-iscc-r:corydalisgreen.121','corydalisgreen','corydalis green',[218,223,183],'dadfb7',14344119],
['nbs-iscc-r:cossackgreen.125','cossackgreen','cossack green',[74,93,35],'4a5d23',4873507],
['nbs-iscc-r:cossackgreen.127','cossackgreen','cossack green',[81,87,68],'515744',5330756],
['nbs-iscc-r:cossegreen.117','cossegreen','cosse green',[126,159,46],'7e9f2e',8298286],
['nbs-iscc-r:cotingapurple.219','cotingapurple','cotinga purple',[96,47,107],'602f6b',6303595],
['nbs-iscc-r:cotingapurple.223','cotingapurple','cotinga purple',[134,96,142],'86608e',8806542],
['nbs-iscc-r:courgegreen.120','courgegreen','courge green',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-r:courtgray.148','courtgray','court gray',[199,230,215],'c7e6d7',13100759],
['nbs-iscc-r:courtgrey.148','courtgrey','court grey ',[199,230,215],'c7e6d7',13100759],
['nbs-iscc-r:creambuff.86','creambuff','cream buff',[248,222,126],'f8de7e',16309886],
['nbs-iscc-r:creambuff.87','creambuff','cream buff',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-r:creamcolor.86','creamcolor','cream color',[248,222,126],'f8de7e',16309886],
['nbs-iscc-r:cressgreen.120','cressgreen','cress green',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-r:cressgreen.125','cressgreen','cress green',[74,93,35],'4a5d23',4873507],
['nbs-iscc-r:cyanineblue.200','cyanineblue','cyanine blue',[78,81,128],'4e5180',5132672],
['nbs-iscc-r:dahliacarmine.259','dahliacarmine','dahlia carmine',[103,49,71],'673147',6762823],
['nbs-iscc-r:dahliapurple.259','dahliapurple','dahlia purple',[103,49,71],'673147',6762823],
['nbs-iscc-r:daphnepink.248','daphnepink','daphne pink',[222,111,161],'de6fa1',14577569],
['nbs-iscc-r:daphnepink.251','daphnepink','daphne pink',[193,126,145],'c17e91',12680849],
['nbs-iscc-r:daphnered.258','daphnered','daphne red',[168,81,110],'a8516e',11030894],
['nbs-iscc-r:darkamericangreen.137','darkamericangreen','dark american green',[53,94,59],'355e3b',3497531],
['nbs-iscc-r:darkamericangreen.150','darkamericangreen','dark american green',[94,113,106],'5e716a',6189418],
['nbs-iscc-r:darkamericangreen.151','darkamericangreen','dark american green',[58,75,71],'3a4b47',3820359],
['nbs-iscc-r:darkanilineblue.187','darkanilineblue','dark aniline blue',[54,69,79],'36454f',3556687],
['nbs-iscc-r:darkanilineblue.229','darkanilineblue','dark aniline blue',[80,64,77],'50404d',5259341],
['nbs-iscc-r:darkanthraceneviolet.224','darkanthraceneviolet','dark anthracene violet',[86,60,92],'563c5c',5651548],
['nbs-iscc-r:darkanthraceneviolet.229','darkanthraceneviolet','dark anthracene violet',[80,64,77],'50404d',5259341],
['nbs-iscc-r:darkbluishglaucous.144','darkbluishglaucous','dark bluish glaucous',[106,171,142],'6aab8e',6990734],
['nbs-iscc-r:darkbluishglaucous.149','darkbluishglaucous','dark bluish glaucous',[141,163,153],'8da399',9282457],
['nbs-iscc-r:darkbluishgraygreen.150','darkbluishgraygreen','dark bluish gray-green',[94,113,106],'5e716a',6189418],
['nbs-iscc-r:darkbluishgraygreen.151','darkbluishgraygreen','dark bluish gray-green',[58,75,71],'3a4b47',3820359],
['nbs-iscc-r:darkbluishgraygreen.164','darkbluishgraygreen','dark bluish gray-green',[49,120,115],'317873',3242099],
['nbs-iscc-r:darkbluishgraygreen.165','darkbluishgraygreen','dark bluish gray-green',[0,75,73],'004b49',19273],
['nbs-iscc-r:darkbluishviolet.211','darkbluishviolet','dark bluish violet',[96,78,129],'604e81',6311553],
['nbs-iscc-r:darkbluishviolet.212','darkbluishviolet','dark bluish violet',[47,33,64],'2f2140',3088704],
['nbs-iscc-r:darkbluishviolet.215','darkbluishviolet','dark bluish violet',[85,76,105],'554c69',5590121],
['nbs-iscc-r:darkcadetblue.182','darkcadetblue','dark cadet blue',[67,107,149],'436b95',4418453],
['nbs-iscc-r:darkcadetblue.204','darkcadetblue','dark cadet blue',[76,81,109],'4c516d',5001581],
['nbs-iscc-r:darkchessyliteblue.174','darkchessyliteblue','dark chessylite blue',[0,73,88],'004958',18776],
['nbs-iscc-r:darkchessyliteblue.182','darkchessyliteblue','dark chessylite blue',[67,107,149],'436b95',4418453],
['nbs-iscc-r:darkchessyliteblue.183','darkchessyliteblue','dark chessylite blue',[0,48,78],'00304e',12366],
['nbs-iscc-r:darkchessyliteblue.186','darkchessyliteblue','dark chessylite blue',[83,104,120],'536878',5466232],
['nbs-iscc-r:darkcinnabargreen.164','darkcinnabargreen','dark cinnabar green',[49,120,115],'317873',3242099],
['nbs-iscc-r:darkcitrine.107','darkcitrine','dark citrine',[102,93,30],'665d1e',6708510],
['nbs-iscc-r:darkcorinthianpurple.259','darkcorinthianpurple','dark corinthian purple',[103,49,71],'673147',6762823],
['nbs-iscc-r:darkcressgreen.125','darkcressgreen','dark cress green',[74,93,35],'4a5d23',4873507],
['nbs-iscc-r:darkdelftblue.183','darkdelftblue','dark delft blue',[0,48,78],'00304e',12366],
['nbs-iscc-r:darkdelftblue.187','darkdelftblue','dark delft blue',[54,69,79],'36454f',3556687],
['nbs-iscc-r:darkdivablue.196','darkdivablue','dark diva blue',[84,90,167],'545aa7',5528231],
['nbs-iscc-r:darkdivablue.200','darkdivablue','dark diva blue',[78,81,128],'4e5180',5132672],
['nbs-iscc-r:darkdullbluishviolet.211','darkdullbluishviolet','dark dull bluish violet',[96,78,129],'604e81',6311553],
['nbs-iscc-r:darkdullbluishviolet.215','darkdullbluishviolet','dark dull bluish violet',[85,76,105],'554c69',5590121],
['nbs-iscc-r:darkdullbluishviolet.215','darkdullbluishviolet','dark dull bluish violet',[85,76,105],'554c69',5590121],
['nbs-iscc-r:darkdullbluishviolet.224','darkdullbluishviolet','dark dull bluish violet',[86,60,92],'563c5c',5651548],
['nbs-iscc-r:darkdullvioletblue.204','darkdullvioletblue','dark dull violet-blue',[76,81,109],'4c516d',5001581],
['nbs-iscc-r:darkdullvioletblue.215','darkdullvioletblue','dark dull violet-blue',[85,76,105],'554c69',5590121],
['nbs-iscc-r:darkdullvioletblue.215','darkdullvioletblue','dark dull violet-blue',[85,76,105],'554c69',5590121],
['nbs-iscc-r:darkdullyellowgreen.125','darkdullyellowgreen','dark dull yellow-green',[74,93,35],'4a5d23',4873507],
['nbs-iscc-r:darkdullyellowgreen.127','darkdullyellowgreen','dark dull yellow-green',[81,87,68],'515744',5330756],
['nbs-iscc-r:darkglaucousgray.150','darkglaucousgray','dark glaucous-gray',[94,113,106],'5e716a',6189418],
['nbs-iscc-r:darkgrayishbluegreen.156','darkgrayishbluegreen','dark grayish blue-green',[78,87,85],'4e5755',5134165],
['nbs-iscc-r:darkgrayishbluegreen.266','darkgrayishbluegreen','dark grayish blue-green',[85,85,85],'555555',5592405],
['nbs-iscc-r:darkgrayishblueviolet.215','darkgrayishblueviolet','dark grayish blue-violet',[85,76,105],'554c69',5590121],
['nbs-iscc-r:darkgrayishbrown.19','darkgrayishbrown','dark grayish brown',[144,93,93],'905d5d',9461085],
['nbs-iscc-r:darkgrayishbrown.23','darkgrayishbrown','dark grayish brown',[92,80,79],'5c504f',6049871],
['nbs-iscc-r:darkgrayishlavender.214','darkgrayishlavender','dark grayish lavender',[150,144,171],'9690ab',9867435],
['nbs-iscc-r:darkgrayisholive.113','darkgrayisholive','dark grayish olive',[87,85,76],'57554c',5723468],
['nbs-iscc-r:darkgreen.137','darkgreen','dark green',[53,94,59],'355e3b',3497531],
['nbs-iscc-r:darkgreen.150','darkgreen','dark green',[94,113,106],'5e716a',6189418],
['nbs-iscc-r:darkgreen.151','darkgreen','dark green',[58,75,71],'3a4b47',3820359],
['nbs-iscc-r:darkgreenbluegray.186','darkgreenbluegray','dark green-blue gray',[83,104,120],'536878',5466232],
['nbs-iscc-r:darkgreenblueslate.183','darkgreenblueslate','dark green-blue slate',[0,48,78],'00304e',12366],
['nbs-iscc-r:darkgreenblueslate.187','darkgreenblueslate','dark green-blue slate',[54,69,79],'36454f',3556687],
['nbs-iscc-r:darkgreenishglaucous.135','darkgreenishglaucous','dark greenish glaucous',[147,197,146],'93c592',9684370],
['nbs-iscc-r:darkgreenishglaucous.136','darkgreenishglaucous','dark greenish glaucous',[103,146,103],'679267',6787687],
['nbs-iscc-r:darkgreenisholive.107','darkgreenisholive','dark greenish olive',[102,93,30],'665d1e',6708510],
['nbs-iscc-r:darkgreenisholive.110','darkgreenisholive','dark greenish olive',[91,88,66],'5b5842',5986370],
['nbs-iscc-r:darkgullgray.266','darkgullgray','dark gull gray',[85,85,85],'555555',5592405],
['nbs-iscc-r:darkheliotropegray.228','darkheliotropegray','dark heliotrope gray',[121,104,120],'796878',7956600],
['nbs-iscc-r:darkheliotropeslate.229','darkheliotropeslate','dark heliotrope slate',[80,64,77],'50404d',5259341],
['nbs-iscc-r:darkheliotropeslate.234','darkheliotropeslate','dark heliotrope slate',[93,85,91],'5d555b',6116699],
['nbs-iscc-r:darkhyssopviolet.224','darkhyssopviolet','dark hyssop violet',[86,60,92],'563c5c',5651548],
['nbs-iscc-r:darkhyssopviolet.228','darkhyssopviolet','dark hyssop violet',[121,104,120],'796878',7956600],
['nbs-iscc-r:darkindianred.19','darkindianred','dark indian red',[144,93,93],'905d5d',9461085],
['nbs-iscc-r:darkindianred.20','darkindianred','dark indian red',[84,61,63],'543d3f',5520703],
['nbs-iscc-r:darkivygreen.127','darkivygreen','dark ivy green',[81,87,68],'515744',5330756],
['nbs-iscc-r:darklavender.222','darklavender','dark lavender',[182,149,192],'b695c0',11965888],
['nbs-iscc-r:darklividbrown.19','darklividbrown','dark livid brown',[144,93,93],'905d5d',9461085],
['nbs-iscc-r:darklividpurple.229','darklividpurple','dark livid purple',[80,64,77],'50404d',5259341],
['nbs-iscc-r:darklividpurple.259','darklividpurple','dark livid purple',[103,49,71],'673147',6762823],
['nbs-iscc-r:darkmadderblue.186','darkmadderblue','dark madder blue',[83,104,120],'536878',5466232],
['nbs-iscc-r:darkmadderblue.204','darkmadderblue','dark madder blue',[76,81,109],'4c516d',5001581],
['nbs-iscc-r:darkmadderviolet.224','darkmadderviolet','dark madder violet',[86,60,92],'563c5c',5651548],
['nbs-iscc-r:darkmadderviolet.229','darkmadderviolet','dark madder violet',[80,64,77],'50404d',5259341],
['nbs-iscc-r:darkmaroonpurple.242','darkmaroonpurple','dark maroon purple',[93,57,84],'5d3954',6109524],
['nbs-iscc-r:darkmediciblue.192','darkmediciblue','dark medici blue',[81,88,94],'51585e',5331038],
['nbs-iscc-r:darkmineralred.19','darkmineralred','dark mineral red ',[144,93,93],'905d5d',9461085],
['nbs-iscc-r:darkmineralred.20','darkmineralred','dark mineral red ',[84,61,63],'543d3f',5520703],
['nbs-iscc-r:darkmousegray.64','darkmousegray','dark mouse gray ',[91,80,79],'5b504f',5984335],
['nbs-iscc-r:darkmousegray.266','darkmousegray','dark mouse gray ',[85,85,85],'555555',5592405],
['nbs-iscc-r:darknaphthaleneviolet.229','darknaphthaleneviolet','dark naphthalene violet ',[80,64,77],'50404d',5259341],
['nbs-iscc-r:darknaphthaleneviolet.242','darknaphthaleneviolet','dark naphthalene violet ',[93,57,84],'5d3954',6109524],
['nbs-iscc-r:darkneutralgray.266','darkneutralgray','dark neutral gray',[85,85,85],'555555',5592405],
['nbs-iscc-r:darknigrosinviolet.224','darknigrosinviolet','dark nigrosin violet',[86,60,92],'563c5c',5651548],
['nbs-iscc-r:darkolive.110','darkolive','dark olive',[91,88,66],'5b5842',5986370],
['nbs-iscc-r:darkolivebuff.91','darkolivebuff','dark olive-buff',[161,143,96],'a18f60',10588000],
['nbs-iscc-r:darkolivegray.110','darkolivegray','dark olive-gray',[91,88,66],'5b5842',5986370],
['nbs-iscc-r:darkolivegray.113','darkolivegray','dark olive-gray',[87,85,76],'57554c',5723468],
['nbs-iscc-r:darkorientblue.186','darkorientblue','dark orient blue',[83,104,120],'536878',5466232],
['nbs-iscc-r:darkpaynesgray.187','darkpaynesgray','dark payne\'s gray',[54,69,79],'36454f',3556687],
['nbs-iscc-r:darkpaynesgray.192','darkpaynesgray','dark payne\'s gray',[81,88,94],'51585e',5331038],
['nbs-iscc-r:darkperillapurple.259','darkperillapurple','dark perilla purple',[103,49,71],'673147',6762823],
['nbs-iscc-r:darkplumbagoblue.214','darkplumbagoblue','dark plumbago blue',[150,144,171],'9690ab',9867435],
['nbs-iscc-r:darkplumbagoblue.227','darkplumbagoblue','dark plumbago blue',[170,152,169],'aa98a9',11180201],
['nbs-iscc-r:darkplumbagogray.228','darkplumbagogray','dark plumbago gray',[121,104,120],'796878',7956600],
['nbs-iscc-r:darkplumbagogray.233','darkplumbagogray','dark plumbago gray',[139,133,137],'8b8589',9143689],
['nbs-iscc-r:darkplumbagogray.234','darkplumbagogray','dark plumbago gray',[93,85,91],'5d555b',6116699],
['nbs-iscc-r:darkplumbagoslate.234','darkplumbagoslate','dark plumbago slate',[93,85,91],'5d555b',6116699],
['nbs-iscc-r:darkplumbeous.192','darkplumbeous','dark plumbeous',[81,88,94],'51585e',5331038],
['nbs-iscc-r:darkporcelaingreen.150','darkporcelaingreen','dark porcelain green',[94,113,106],'5e716a',6189418],
['nbs-iscc-r:darkpurpledrab.228','darkpurpledrab','dark purple-drab',[121,104,120],'796878',7956600],
['nbs-iscc-r:darkpurplishgray.234','darkpurplishgray','dark purplish gray',[93,85,91],'5d555b',6116699],
['nbs-iscc-r:darkpurplishgray.266','darkpurplishgray','dark purplish gray',[85,85,85],'555555',5592405],
['nbs-iscc-r:darkquakerdrab.234','darkquakerdrab','dark quaker drab',[93,85,91],'5d555b',6116699],
['nbs-iscc-r:darkrussiangreen.150','darkrussiangreen','dark russian green',[94,113,106],'5e716a',6189418],
['nbs-iscc-r:darkrussiangreen.151','darkrussiangreen','dark russian green',[58,75,71],'3a4b47',3820359],
['nbs-iscc-r:darkslatepurple.242','darkslatepurple','dark slate-purple',[93,57,84],'5d3954',6109524],
['nbs-iscc-r:darkslateviolet.234','darkslateviolet','dark slate-violet ',[93,85,91],'5d555b',6116699],
['nbs-iscc-r:darkslateviolet.228','darkslateviolet','dark slate-violet ',[121,104,120],'796878',7956600],
['nbs-iscc-r:darksoftblueviolet.200','darksoftblueviolet','dark soft blue-violet',[78,81,128],'4e5180',5132672],
['nbs-iscc-r:darksoftblueviolet.211','darksoftblueviolet','dark soft blue-violet',[96,78,129],'604e81',6311553],
['nbs-iscc-r:darksoftbluishviolet.211','darksoftbluishviolet','dark soft bluish violet',[96,78,129],'604e81',6311553],
['nbs-iscc-r:darksulphategreen.164','darksulphategreen','dark sulphate green',[49,120,115],'317873',3242099],
['nbs-iscc-r:darkterreverte.150','darkterreverte','dark terre verte',[94,113,106],'5e716a',6189418],
['nbs-iscc-r:darktyrianblue.186','darktyrianblue','dark tyrian blue',[83,104,120],'536878',5466232],
['nbs-iscc-r:darktyrianblue.204','darktyrianblue','dark tyrian blue',[76,81,109],'4c516d',5001581],
['nbs-iscc-r:darkvarleysgray.234','darkvarleysgray','dark varley\'s gray',[93,85,91],'5d555b',6116699],
['nbs-iscc-r:darkvinaceous.19','darkvinaceous','dark vinaceous',[144,93,93],'905d5d',9461085],
['nbs-iscc-r:darkvinaceousbrown.19','darkvinaceousbrown','dark vinaceous-brown',[144,93,93],'905d5d',9461085],
['nbs-iscc-r:darkvinaceousdrab.19','darkvinaceousdrab','dark vinaceous-drab',[144,93,93],'905d5d',9461085],
['nbs-iscc-r:darkvinaceousdrab.23','darkvinaceousdrab','dark vinaceous-drab',[92,80,79],'5c504f',6049871],
['nbs-iscc-r:darkvinaceousgray.228','darkvinaceousgray','dark vinaceous-gray',[121,104,120],'796878',7956600],
['nbs-iscc-r:darkvinaceouspurple.259','darkvinaceouspurple','dark vinaceous-purple',[103,49,71],'673147',6762823],
['nbs-iscc-r:darkviolet.211','darkviolet','dark violet',[96,78,129],'604e81',6311553],
['nbs-iscc-r:darkvioletgray.234','darkvioletgray','dark violet-gray',[93,85,91],'5d555b',6116699],
['nbs-iscc-r:darkvioletgray.266','darkvioletgray','dark violet-gray',[85,85,85],'555555',5592405],
['nbs-iscc-r:darkvioletslate.186','darkvioletslate','dark violet-slate',[83,104,120],'536878',5466232],
['nbs-iscc-r:darkviridiangreen.145','darkviridiangreen','dark viridian green',[59,120,97],'3b7861',3897441],
['nbs-iscc-r:darkyellowishgreen.137','darkyellowishgreen','dark yellowish green',[53,94,59],'355e3b',3497531],
['nbs-iscc-r:darkyvetteviolet.229','darkyvetteviolet','dark yvette violet',[80,64,77],'50404d',5259341],
['nbs-iscc-r:darkzincgreen.145','darkzincgreen','dark zinc green',[59,120,97],'3b7861',3897441],
['nbs-iscc-r:dauphinsviolet.211','dauphinsviolet','dauphin\'s violet',[96,78,129],'604e81',6311553],
['nbs-iscc-r:dawngray.190','dawngray','dawn gray',[180,188,192],'b4bcc0',11844800],
['nbs-iscc-r:dawngray.264','dawngray','dawn gray',[185,184,181],'b9b8b5',12171445],
['nbs-iscc-r:deepanilinelilac.177','deepanilinelilac','deep aniline lilac',[73,151,208],'4997d0',4822992],
['nbs-iscc-r:deepblueviolet.194','deepblueviolet','deep blue-violet',[48,38,122],'30267a',3155578],
['nbs-iscc-r:deepblueviolet.205','deepblueviolet','deep blue-violet',[144,101,202],'9065ca',9463242],
['nbs-iscc-r:deepbluishglaucous.162','deepbluishglaucous','deep bluish glaucous',[150,222,209],'96ded1',9887441],
['nbs-iscc-r:deepbluishgraygreen.150','deepbluishgraygreen','deep bluish gray-green',[94,113,106],'5e716a',6189418],
['nbs-iscc-r:deepbluishgraygreen.164','deepbluishgraygreen','deep bluish gray-green',[49,120,115],'317873',3242099],
['nbs-iscc-r:deepbrownishdrab.61','deepbrownishdrab','deep brownish drab',[99,81,71],'635147',6508871],
['nbs-iscc-r:deepbrownishvinaceous.19','deepbrownishvinaceous','deep brownish vinaceous',[144,93,93],'905d5d',9461085],
['nbs-iscc-r:deepcadetblue.182','deepcadetblue','deep cadet blue',[67,107,149],'436b95',4418453],
['nbs-iscc-r:deepchicoryblue.206','deepchicoryblue','deep chicory blue',[126,115,184],'7e73b8',8287160],
['nbs-iscc-r:deepchrome.68','deepchrome','deep chrome',[234,162,33],'eaa221',15376929],
['nbs-iscc-r:deepchrysolitegreen.120','deepchrysolitegreen','deep chrysolite green',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-r:deepcolonialbuff.87','deepcolonialbuff','deep colonial buff',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-r:deepcorinthianred.19','deepcorinthianred','deep corinthian red',[144,93,93],'905d5d',9461085],
['nbs-iscc-r:deepdelftblue.186','deepdelftblue','deep delft blue',[83,104,120],'536878',5466232],
['nbs-iscc-r:deepdullbluishviolet.207','deepdullbluishviolet','deep dull bluish violet',[96,78,151],'604e97',6311575],
['nbs-iscc-r:deepdullbluishviolet.211','deepdullbluishviolet','deep dull bluish violet',[96,78,129],'604e81',6311553],
['nbs-iscc-r:deepdullbluishviolet.203','deepdullbluishviolet','deep dull bluish violet',[140,146,172],'8c92ac',9212588],
['nbs-iscc-r:deepdullbluishviolet.204','deepdullbluishviolet','deep dull bluish violet',[76,81,109],'4c516d',5001581],
['nbs-iscc-r:deepdullbluishviolet.214','deepdullbluishviolet','deep dull bluish violet',[150,144,171],'9690ab',9867435],
['nbs-iscc-r:deepdullbluishviolet.215','deepdullbluishviolet','deep dull bluish violet',[85,76,105],'554c69',5590121],
['nbs-iscc-r:deepdullbluishviolet.211','deepdullbluishviolet','deep dull bluish violet',[96,78,129],'604e81',6311553],
['nbs-iscc-r:deepdulllavender.227','deepdulllavender','deep dull lavender',[170,152,169],'aa98a9',11180201],
['nbs-iscc-r:deepdullviolaceousblue.200','deepdullviolaceousblue','deep dull violaceous blue',[78,81,128],'4e5180',5132672],
['nbs-iscc-r:deepdullvioletblue.211','deepdullvioletblue','deep dull violet-blue',[96,78,129],'604e81',6311553],
['nbs-iscc-r:deepdutchblue.186','deepdutchblue','deep dutch blue',[83,104,120],'536878',5466232],
['nbs-iscc-r:deepdutchblue.203','deepdutchblue','deep dutch blue',[140,146,172],'8c92ac',9212588],
['nbs-iscc-r:deepglaucousgray.191','deepglaucousgray','deep glaucous-gray',[129,135,139],'81878b',8488843],
['nbs-iscc-r:deepglaucousgreen.144','deepglaucousgreen','deep glaucous-green',[106,171,142],'6aab8e',6990734],
['nbs-iscc-r:deepgrapegreen.120','deepgrapegreen','deep grape green',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-r:deepgrapegreen.122','deepgrapegreen','deep grape green',[143,151,121],'8f9779',9410425],
['nbs-iscc-r:deepgrayishbluegreen.150','deepgrayishbluegreen','deep grayish blue-green',[94,113,106],'5e716a',6189418],
['nbs-iscc-r:deepgrayishlavender.214','deepgrayishlavender','deep grayish lavender ',[150,144,171],'9690ab',9867435],
['nbs-iscc-r:deepgrayishlavender.227','deepgrayishlavender','deep grayish lavender ',[170,152,169],'aa98a9',11180201],
['nbs-iscc-r:deepgrayisholive.113','deepgrayisholive','deep grayish olive',[87,85,76],'57554c',5723468],
['nbs-iscc-r:deepgreenbluegray.186','deepgreenbluegray','deep green-blue gray',[83,104,120],'536878',5466232],
['nbs-iscc-r:deepgreenishglaucous.143','deepgreenishglaucous','deep greenish glaucous',[142,209,178],'8ed1b2',9359794],
['nbs-iscc-r:deepgullgray.265','deepgullgray','deep gull gray',[132,132,130],'848482',8684674],
['nbs-iscc-r:deepheliotropegray.228','deepheliotropegray','deep heliotrope gray ',[121,104,120],'796878',7956600],
['nbs-iscc-r:deephyssopviolet.223','deephyssopviolet','deep hyssop violet',[134,96,142],'86608e',8806542],
['nbs-iscc-r:deeplavender.210','deeplavender','deep lavender',[140,130,182],'8c82b6',9208502],
['nbs-iscc-r:deeplavenderblue.195','deeplavenderblue','deep lavender-blue',[108,121,184],'6c79b8',7109048],
['nbs-iscc-r:deeplichengreen.135','deeplichengreen','deep lichen green',[147,197,146],'93c592',9684370],
['nbs-iscc-r:deeplividbrown.19','deeplividbrown','deep livid brown',[144,93,93],'905d5d',9461085],
['nbs-iscc-r:deeplividpurple.259','deeplividpurple','deep livid purple',[103,49,71],'673147',6762823],
['nbs-iscc-r:deepmadderblue.204','deepmadderblue','deep madder blue',[76,81,109],'4c516d',5001581],
['nbs-iscc-r:deepmalachitegreen.136','deepmalachitegreen','deep malachite green',[103,146,103],'679267',6787687],
['nbs-iscc-r:deepmediciblue.191','deepmediciblue','deep medici blue',[129,135,139],'81878b',8488843],
['nbs-iscc-r:deepmousegray.64','deepmousegray','deep mouse gray ',[91,80,79],'5b504f',5984335],
['nbs-iscc-r:deepmousegray.265','deepmousegray','deep mouse gray ',[132,132,130],'848482',8684674],
['nbs-iscc-r:deepmousegray.266','deepmousegray','deep mouse gray ',[85,85,85],'555555',5592405],
['nbs-iscc-r:deepneutralgray.266','deepneutralgray','deep neutral gray',[85,85,85],'555555',5592405],
['nbs-iscc-r:deepolive.107','deepolive','deep olive',[102,93,30],'665d1e',6708510],
['nbs-iscc-r:deepolive.110','deepolive','deep olive',[91,88,66],'5b5842',5986370],
['nbs-iscc-r:deepolivebuff.90','deepolivebuff','deep olive-buff',[194,178,128],'c2b280',12759680],
['nbs-iscc-r:deepolivegray.112','deepolivegray','deep olive-gray',[138,135,118],'8a8776',9078646],
['nbs-iscc-r:deeporientblue.186','deeporientblue','deep orient blue',[83,104,120],'536878',5466232],
['nbs-iscc-r:deeppaynesgray.186','deeppaynesgray','deep payne\'s gray',[83,104,120],'536878',5466232],
['nbs-iscc-r:deepplumbagoblue.227','deepplumbagoblue','deep plumbago blue',[170,152,169],'aa98a9',11180201],
['nbs-iscc-r:deepplumbagogray.233','deepplumbagogray','deep plumbago gray',[139,133,137],'8b8589',9143689],
['nbs-iscc-r:deepplumbeous.186','deepplumbeous','deep plumbeous',[83,104,120],'536878',5466232],
['nbs-iscc-r:deepplumbeous.191','deepplumbeous','deep plumbeous',[129,135,139],'81878b',8488843],
['nbs-iscc-r:deeppurplishgray.234','deeppurplishgray','deep purplish gray',[93,85,91],'5d555b',6116699],
['nbs-iscc-r:deeppurplishvinaceous.262','deeppurplishvinaceous','deep purplish vinaceous',[145,95,109],'915f6d',9527149],
['nbs-iscc-r:deepquakerdrab.234','deepquakerdrab','deep quaker drab',[93,85,91],'5d555b',6116699],
['nbs-iscc-r:deeprosepink.247','deeprosepink','deep rose pink',[230,143,172],'e68fac',15110060],
['nbs-iscc-r:deepseafoamgreen.119','deepseafoamgreen','deep sea-foam green ',[201,220,137],'c9dc89',13229193],
['nbs-iscc-r:deepslateblue.186','deepslateblue','deep slate-blue',[83,104,120],'536878',5466232],
['nbs-iscc-r:deepslategreen.156','deepslategreen','deep slate-green',[78,87,85],'4e5755',5134165],
['nbs-iscc-r:deepslateolive.156','deepslateolive','deep slate-olive',[78,87,85],'4e5755',5134165],
['nbs-iscc-r:deepslateviolet.228','deepslateviolet','deep slate-violet',[121,104,120],'796878',7956600],
['nbs-iscc-r:deepslatybrown.229','deepslatybrown','deep slaty brown',[80,64,77],'50404d',5259341],
['nbs-iscc-r:deepsoftblueviolet.200','deepsoftblueviolet','deep soft blue-violet',[78,81,128],'4e5180',5132672],
['nbs-iscc-r:deepsoftblueviolet.211','deepsoftblueviolet','deep soft blue-violet',[96,78,129],'604e81',6311553],
['nbs-iscc-r:deepsoftbluishviolet.207','deepsoftbluishviolet','deep soft bluish violet',[96,78,151],'604e97',6311575],
['nbs-iscc-r:deepturtlegreen.131','deepturtlegreen','deep turtle green',[68,148,74],'44944a',4494410],
['nbs-iscc-r:deepturtlegreen.136','deepturtlegreen','deep turtle green',[103,146,103],'679267',6787687],
['nbs-iscc-r:deepvarleysgray.228','deepvarleysgray','deep varley\'s gray',[121,104,120],'796878',7956600],
['nbs-iscc-r:deepvarleysgray.234','deepvarleysgray','deep varley\'s gray',[93,85,91],'5d555b',6116699],
['nbs-iscc-r:deepvinaceous.6','deepvinaceous','deep vinaceous',[192,128,129],'c08081',12615809],
['nbs-iscc-r:deepvinaceousgray.228','deepvinaceousgray','deep vinaceous-gray',[121,104,120],'796878',7956600],
['nbs-iscc-r:deepvinaceouslavender.244','deepvinaceouslavender','deep vinaceous-lavender',[170,138,158],'aa8a9e',11176606],
['nbs-iscc-r:deepvinaceouslavender.253','deepvinaceouslavender','deep vinaceous-lavender',[195,166,177],'c3a6b1',12822193],
['nbs-iscc-r:deepvioletgray.233','deepvioletgray','deep violet-gray',[139,133,137],'8b8589',9143689],
['nbs-iscc-r:deepvioletgray.234','deepvioletgray','deep violet-gray',[93,85,91],'5d555b',6116699],
['nbs-iscc-r:deepvioletgray.265','deepvioletgray','deep violet-gray',[132,132,130],'848482',8684674],
['nbs-iscc-r:deepvioletgray.266','deepvioletgray','deep violet-gray',[85,85,85],'555555',5592405],
['nbs-iscc-r:deepvioletplumbeous.204','deepvioletplumbeous','deep violet-plumbeous ',[76,81,109],'4c516d',5001581],
['nbs-iscc-r:deepwedgewoodblue.195','deepwedgewoodblue','deep wedgewood blue',[108,121,184],'6c79b8',7109048],
['nbs-iscc-r:delftblue.186','delftblue','delft blue',[83,104,120],'536878',5466232],
['nbs-iscc-r:diaminazoblue.215','diaminazoblue','diamin-azo blue',[85,76,105],'554c69',5590121],
['nbs-iscc-r:diaminazoblue.229','diaminazoblue','diamin-azo blue',[80,64,77],'50404d',5259341],
['nbs-iscc-r:diaminebrown.46','diaminebrown','diamine brown',[103,76,71],'674c47',6769735],
['nbs-iscc-r:diaminegreen.145','diaminegreen','diamine green',[59,120,97],'3b7861',3897441],
['nbs-iscc-r:diaminegreen.146','diaminegreen','diamine green',[27,77,62],'1b4d3e',1789246],
['nbs-iscc-r:divablue.195','divablue','diva blue',[108,121,184],'6c79b8',7109048],
['nbs-iscc-r:divablue.196','divablue','diva blue',[84,90,167],'545aa7',5528231],
['nbs-iscc-r:drab.76','drab','drab',[193,154,107],'c19a6b',12687979],
['nbs-iscc-r:drab.77','drab','drab',[130,102,68],'826644',8545860],
['nbs-iscc-r:drab.79','drab','drab',[174,155,130],'ae9b82',11443074],
['nbs-iscc-r:drab.80','drab','drab',[126,109,90],'7e6d5a',8285530],
['nbs-iscc-r:drabgray.79','drabgray','drab-gray',[174,155,130],'ae9b82',11443074],
['nbs-iscc-r:drabgray.93','drabgray','drab-gray',[191,184,165],'bfb8a5',12564645],
['nbs-iscc-r:dragonsbloodred.38','dragonsbloodred','dragons-blood red',[158,71,50],'9e4732',10372914],
['nbs-iscc-r:dresdenbrown.77','dresdenbrown','dresden brown',[130,102,68],'826644',8545860],
['nbs-iscc-r:duckgreen.151','duckgreen','duck green',[58,75,71],'3a4b47',3820359],
['nbs-iscc-r:dullblackishgreen.151','dullblackishgreen','dull blackish green',[58,75,71],'3a4b47',3820359],
['nbs-iscc-r:dullbluegreenblack.187','dullbluegreenblack','dull blue-green black',[54,69,79],'36454f',3556687],
['nbs-iscc-r:dullbluegreenblack.188','dullbluegreenblack','dull blue-green black',[32,40,48],'202830',2107440],
['nbs-iscc-r:dullbluegreenblack.193','dullbluegreenblack','dull blue-green black',[32,36,40],'202428',2106408],
['nbs-iscc-r:dullblueviolet.196','dullblueviolet','dull blue-violet',[84,90,167],'545aa7',5528231],
['nbs-iscc-r:dullblueviolet.206','dullblueviolet','dull blue-violet',[126,115,184],'7e73b8',8287160],
['nbs-iscc-r:dullblueviolet.207','dullblueviolet','dull blue-violet',[96,78,151],'604e97',6311575],
['nbs-iscc-r:dullblueviolet.211','dullblueviolet','dull blue-violet',[96,78,129],'604e81',6311553],
['nbs-iscc-r:dullbluishviolet.206','dullbluishviolet','dull bluish violet',[126,115,184],'7e73b8',8287160],
['nbs-iscc-r:dullbluishviolet.207','dullbluishviolet','dull bluish violet',[96,78,151],'604e97',6311575],
['nbs-iscc-r:dullbluishviolet.210','dullbluishviolet','dull bluish violet',[140,130,182],'8c82b6',9208502],
['nbs-iscc-r:dullbluishviolet.211','dullbluishviolet','dull bluish violet',[96,78,129],'604e81',6311553],
['nbs-iscc-r:dullbluishviolet.207','dullbluishviolet','dull bluish violet',[96,78,151],'604e97',6311575],
['nbs-iscc-r:dullbluishviolet.211','dullbluishviolet','dull bluish violet',[96,78,129],'604e81',6311553],
['nbs-iscc-r:dullcitrine.106','dullcitrine','dull citrine',[134,126,54],'867e36',8814134],
['nbs-iscc-r:dullcitrine.107','dullcitrine','dull citrine',[102,93,30],'665d1e',6708510],
['nbs-iscc-r:dulldarkpurple.237','dulldarkpurple','dull dark purple',[158,79,136],'9e4f88',10375048],
['nbs-iscc-r:dulldarkpurple.238','dulldarkpurple','dull dark purple',[112,41,99],'702963',7350627],
['nbs-iscc-r:dulldarkpurple.241','dulldarkpurple','dull dark purple',[145,92,131],'915c83',9526403],
['nbs-iscc-r:dullduskypurple.229','dullduskypurple','dull dusky purple',[80,64,77],'50404d',5259341],
['nbs-iscc-r:dullduskypurple.242','dullduskypurple','dull dusky purple',[93,57,84],'5d3954',6109524],
['nbs-iscc-r:dullgreenishblack.156','dullgreenishblack','dull greenish black ',[78,87,85],'4e5755',5134165],
['nbs-iscc-r:dullgreenishblack.266','dullgreenishblack','dull greenish black ',[85,85,85],'555555',5592405],
['nbs-iscc-r:dullgreenishblack.156','dullgreenishblack','dull greenish black ',[78,87,85],'4e5755',5134165],
['nbs-iscc-r:dullgreenishblack.266','dullgreenishblack','dull greenish black ',[85,85,85],'555555',5592405],
['nbs-iscc-r:dullgreenyellow.117','dullgreenyellow','dull green-yellow',[126,159,46],'7e9f2e',8298286],
['nbs-iscc-r:dullindianpurple.262','dullindianpurple','dull indian purple',[145,95,109],'915f6d',9527149],
['nbs-iscc-r:dulllavender.253','dulllavender','dull lavender',[195,166,177],'c3a6b1',12822193],
['nbs-iscc-r:dullmagentapurple.237','dullmagentapurple','dull magenta purple',[158,79,136],'9e4f88',10375048],
['nbs-iscc-r:dullopalinegreen.148','dullopalinegreen','dull opaline green',[199,230,215],'c7e6d7',13100759],
['nbs-iscc-r:dullpurplishblack.234','dullpurplishblack','dull purplish black',[93,85,91],'5d555b',6116699],
['nbs-iscc-r:dullviolaceousblue.196','dullviolaceousblue','dull violaceous blue',[84,90,167],'545aa7',5528231],
['nbs-iscc-r:dullvioletblack.234','dullvioletblack','dull violet-black ',[93,85,91],'5d555b',6116699],
['nbs-iscc-r:dullvioletblack.187','dullvioletblack','dull violet-black ',[54,69,79],'36454f',3556687],
['nbs-iscc-r:dullvioletblack.192','dullvioletblack','dull violet-black ',[81,88,94],'51585e',5331038],
['nbs-iscc-r:dullvioletblack.229','dullvioletblack','dull violet-black ',[80,64,77],'50404d',5259341],
['nbs-iscc-r:dullvioletblack.234','dullvioletblack','dull violet-black ',[93,85,91],'5d555b',6116699],
['nbs-iscc-r:dullvioletblack.266','dullvioletblack','dull violet-black ',[85,85,85],'555555',5592405],
['nbs-iscc-r:dullvioletblack.267','dullvioletblack','dull violet-black ',[34,34,34],'222222',2236962],
['nbs-iscc-r:dullvioletblue.196','dullvioletblue','dull violet-blue',[84,90,167],'545aa7',5528231],
['nbs-iscc-r:dullvioletblue.211','dullvioletblue','dull violet-blue',[96,78,129],'604e81',6311553],
['nbs-iscc-r:duskyauriculapurple.242','duskyauriculapurple','dusky auricula purple',[93,57,84],'5d3954',6109524],
['nbs-iscc-r:duskyblue.204','duskyblue','dusky blue',[76,81,109],'4c516d',5001581],
['nbs-iscc-r:duskybluegreen.151','duskybluegreen','dusky blue-green',[58,75,71],'3a4b47',3820359],
['nbs-iscc-r:duskyblueviolet.215','duskyblueviolet','dusky blue-violet',[85,76,105],'554c69',5590121],
['nbs-iscc-r:duskyblueviolet.212','duskyblueviolet','dusky blue-violet',[47,33,64],'2f2140',3088704],
['nbs-iscc-r:duskyblueviolet.229','duskyblueviolet','dusky blue-violet',[80,64,77],'50404d',5259341],
['nbs-iscc-r:duskybluishgreen.156','duskybluishgreen','dusky bluish green',[78,87,85],'4e5755',5134165],
['nbs-iscc-r:duskybrown.229','duskybrown','dusky brown ',[80,64,77],'50404d',5259341],
['nbs-iscc-r:duskybrown.234','duskybrown','dusky brown ',[93,85,91],'5d555b',6116699],
['nbs-iscc-r:duskydrab.64','duskydrab','dusky drab',[91,80,79],'5b504f',5984335],
['nbs-iscc-r:duskydullbluishgreen.151','duskydullbluishgreen','dusky dull bluish green',[58,75,71],'3a4b47',3820359],
['nbs-iscc-r:duskydullgreen.151','duskydullgreen','dusky dull green',[58,75,71],'3a4b47',3820359],
['nbs-iscc-r:duskydullgreen.156','duskydullgreen','dusky dull green',[78,87,85],'4e5755',5134165],
['nbs-iscc-r:duskydullviolet.224','duskydullviolet','dusky dull violet ',[86,60,92],'563c5c',5651548],
['nbs-iscc-r:duskydullviolet.223','duskydullviolet','dusky dull violet ',[134,96,142],'86608e',8806542],
['nbs-iscc-r:duskydullviolet.224','duskydullviolet','dusky dull violet ',[86,60,92],'563c5c',5651548],
['nbs-iscc-r:duskydullviolet.228','duskydullviolet','dusky dull violet ',[121,104,120],'796878',7956600],
['nbs-iscc-r:duskydullvioletblue.215','duskydullvioletblue','dusky dull violet-blue ',[85,76,105],'554c69',5590121],
['nbs-iscc-r:duskygreen.150','duskygreen','dusky green',[94,113,106],'5e716a',6189418],
['nbs-iscc-r:duskygreen.151','duskygreen','dusky green',[58,75,71],'3a4b47',3820359],
['nbs-iscc-r:duskygreenblue.165','duskygreenblue','dusky green-blue ',[0,75,73],'004b49',19273],
['nbs-iscc-r:duskygreenblue.174','duskygreenblue','dusky green-blue ',[0,73,88],'004958',18776],
['nbs-iscc-r:duskygreenblue.186','duskygreenblue','dusky green-blue ',[83,104,120],'536878',5466232],
['nbs-iscc-r:duskygreenblue.186','duskygreenblue','dusky green-blue ',[83,104,120],'536878',5466232],
['nbs-iscc-r:duskygreenblue.187','duskygreenblue','dusky green-blue ',[54,69,79],'36454f',3556687],
['nbs-iscc-r:duskygreengray.266','duskygreengray','dusky green-gray',[85,85,85],'555555',5592405],
['nbs-iscc-r:duskygreenishblue.179','duskygreenishblue','dusky greenish blue',[0,65,106],'00416a',16746],
['nbs-iscc-r:duskygreenishblue.182','duskygreenishblue','dusky greenish blue',[67,107,149],'436b95',4418453],
['nbs-iscc-r:duskygreenishblue.183','duskygreenishblue','dusky greenish blue',[0,48,78],'00304e',12366],
['nbs-iscc-r:duskyneutralgray.266','duskyneutralgray','dusky neutral gray',[85,85,85],'555555',5592405],
['nbs-iscc-r:duskyolivegreen.127','duskyolivegreen','dusky olive-green',[81,87,68],'515744',5330756],
['nbs-iscc-r:duskyorientblue.186','duskyorientblue','dusky orient blue',[83,104,120],'536878',5466232],
['nbs-iscc-r:duskypurplishgray.229','duskypurplishgray','dusky purplish gray',[80,64,77],'50404d',5259341],
['nbs-iscc-r:duskypurplishgray.234','duskypurplishgray','dusky purplish gray',[93,85,91],'5d555b',6116699],
['nbs-iscc-r:duskypurplishgray.266','duskypurplishgray','dusky purplish gray',[85,85,85],'555555',5592405],
['nbs-iscc-r:duskypurplishgray.267','duskypurplishgray','dusky purplish gray',[34,34,34],'222222',2236962],
['nbs-iscc-r:duskyslateblue.187','duskyslateblue','dusky slate-blue',[54,69,79],'36454f',3556687],
['nbs-iscc-r:duskyslateviolet.229','duskyslateviolet','dusky slate-violet',[80,64,77],'50404d',5259341],
['nbs-iscc-r:duskyslateviolet.234','duskyslateviolet','dusky slate-violet',[93,85,91],'5d555b',6116699],
['nbs-iscc-r:duskyviolet.229','duskyviolet','dusky violet',[80,64,77],'50404d',5259341],
['nbs-iscc-r:duskyvioletblue.204','duskyvioletblue','dusky violet-blue ',[76,81,109],'4c516d',5001581],
['nbs-iscc-r:duskyvioletblue.215','duskyvioletblue','dusky violet-blue ',[85,76,105],'554c69',5590121],
['nbs-iscc-r:duskyvioletblue.187','duskyvioletblue','dusky violet-blue ',[54,69,79],'36454f',3556687],
['nbs-iscc-r:duskyyellowishgreen.127','duskyyellowishgreen','dusky yellowish green',[81,87,68],'515744',5330756],
['nbs-iscc-r:dutchblue.185','dutchblue','dutch blue',[145,163,176],'91a3b0',9544624],
['nbs-iscc-r:dutchblue.203','dutchblue','dutch blue',[140,146,172],'8c92ac',9212588],
['nbs-iscc-r:ecrudrab.33','ecrudrab','ecru-drab',[194,172,153],'c2ac99',12758169],
['nbs-iscc-r:ecruolive.88','ecruolive','ecru-olive',[171,145,68],'ab9144',11243844],
['nbs-iscc-r:ecruolive.91','ecruolive','ecru-olive',[161,143,96],'a18f60',10588000],
['nbs-iscc-r:elmgreen.125','elmgreen','elm green',[74,93,35],'4a5d23',4873507],
['nbs-iscc-r:elmgreen.127','elmgreen','elm green',[81,87,68],'515744',5330756],
['nbs-iscc-r:emeraldgreen.130','emeraldgreen','emerald green',[131,211,125],'83d37d',8639357],
['nbs-iscc-r:empiregreen.150','empiregreen','empire green',[94,113,106],'5e716a',6189418],
['nbs-iscc-r:empiregreen.151','empiregreen','empire green',[58,75,71],'3a4b47',3820359],
['nbs-iscc-r:empireyellow.83','empireyellow','empire yellow',[250,218,94],'fada5e',16439902],
['nbs-iscc-r:endiveblue.199','endiveblue','endive blue',[135,145,191],'8791bf',8884671],
['nbs-iscc-r:endiveblue.203','endiveblue','endive blue',[140,146,172],'8c92ac',9212588],
['nbs-iscc-r:englishred.38','englishred','english red',[158,71,50],'9e4732',10372914],
['nbs-iscc-r:eosinepink.3','eosinepink','eosine pink',[228,113,122],'e4717a',14971258],
['nbs-iscc-r:ethylgreen.160','ethylgreen','ethyl green',[0,122,116],'007a74',31348],
['nbs-iscc-r:ethylgreen.164','ethylgreen','ethyl green',[49,120,115],'317873',3242099],
['nbs-iscc-r:etonblue.182','etonblue','eton blue',[67,107,149],'436b95',4418453],
['nbs-iscc-r:etruscanred.19','etruscanred','etruscan red',[144,93,93],'905d5d',9461085],
['nbs-iscc-r:eugeniared.15','eugeniared','eugenia red',[171,78,82],'ab4e52',11226706],
['nbs-iscc-r:eupatoriumpurple.240','eupatoriumpurple','eupatorium purple',[183,132,167],'b784a7',12027047],
['nbs-iscc-r:eupatoriumpurple.241','eupatoriumpurple','eupatorium purple',[145,92,131],'915c83',9526403],
['nbs-iscc-r:fawncolor.45','fawncolor','fawn color',[151,127,115],'977f73',9928563],
['nbs-iscc-r:fawncolor.57','fawncolor','fawn color',[166,123,91],'a67b5b',10910555],
['nbs-iscc-r:fawncolor.60','fawncolor','fawn color',[149,128,112],'958070',9797744],
['nbs-iscc-r:ferruginous.37','ferruginous','ferruginous',[203,109,81],'cb6d51',13331793],
['nbs-iscc-r:ferruginous.38','ferruginous','ferruginous',[158,71,50],'9e4732',10372914],
['nbs-iscc-r:flamescarlet.34','flamescarlet','flame scarlet ',[226,88,34],'e25822',14833698],
['nbs-iscc-r:flaxflowerblue.195','flaxflowerblue','flax-flower blue ',[108,121,184],'6c79b8',7109048],
['nbs-iscc-r:fleshcolor.5','fleshcolor','flesh color',[222,165,164],'dea5a4',14591396],
['nbs-iscc-r:fleshochre.53','fleshochre','flesh-ochre ',[217,144,88],'d99058',14258264],
['nbs-iscc-r:fleshpink.28','fleshpink','flesh pink',[244,194,194],'f4c2c2',16040642],
['nbs-iscc-r:fleshpink.29','fleshpink','flesh pink',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-r:fluoritegreen.136','fluoritegreen','fluorite green',[103,146,103],'679267',6787687],
['nbs-iscc-r:fluoriteviolet.224','fluoriteviolet','fluorite violet',[86,60,92],'563c5c',5651548],
['nbs-iscc-r:forestgreen.120','forestgreen','forest green',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-r:forestgreen.125','forestgreen','forest green',[74,93,35],'4a5d23',4873507],
['nbs-iscc-r:forgetmenotblue.195','forgetmenotblue','forget-me-not blue',[108,121,184],'6c79b8',7109048],
['nbs-iscc-r:frenchgray.185','frenchgray','french gray',[145,163,176],'91a3b0',9544624],
['nbs-iscc-r:frenchgray.190','frenchgray','french gray',[180,188,192],'b4bcc0',11844800],
['nbs-iscc-r:frenchgreen.145','frenchgreen','french green',[59,120,97],'3b7861',3897441],
['nbs-iscc-r:fuscous.64','fuscous','fuscous',[91,80,79],'5b504f',5984335],
['nbs-iscc-r:fuscousblack.64','fuscousblack','fuscous-black',[91,80,79],'5b504f',5984335],
['nbs-iscc-r:fuscousblack.266','fuscousblack','fuscous-black',[85,85,85],'555555',5592405],
['nbs-iscc-r:garnetbrown.13','garnetbrown','garnet brown',[132,27,45],'841b2d',8657709],
['nbs-iscc-r:gendarmeblue.182','gendarmeblue','gendarme blue',[67,107,149],'436b95',4418453],
['nbs-iscc-r:gentianblue.196','gentianblue','gentian blue',[84,90,167],'545aa7',5528231],
['nbs-iscc-r:geraniumpink.3','geraniumpink','geranium pink',[228,113,122],'e4717a',14971258],
['nbs-iscc-r:geraniumpink.27','geraniumpink','geranium pink',[230,103,97],'e66761',15099745],
['nbs-iscc-r:glassgreen.119','glassgreen','glass green',[201,220,137],'c9dc89',13229193],
['nbs-iscc-r:glaucous.121','glaucous','glaucous',[218,223,183],'dadfb7',14344119],
['nbs-iscc-r:glaucousblue.172','glaucousblue','glaucous-blue',[102,170,188],'66aabc',6728380],
['nbs-iscc-r:glaucousgray.190','glaucousgray','glaucous-gray',[180,188,192],'b4bcc0',11844800],
['nbs-iscc-r:glaucousgreen.149','glaucousgreen','glaucous-green',[141,163,153],'8da399',9282457],
['nbs-iscc-r:gnaphaliumgreen.148','gnaphaliumgreen','gnaphalium green',[199,230,215],'c7e6d7',13100759],
['nbs-iscc-r:gnaphaliumgreen.149','gnaphaliumgreen','gnaphalium green',[141,163,153],'8da399',9282457],
['nbs-iscc-r:grassgreen.120','grassgreen','grass green',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-r:grayishbluegreen.150','grayishbluegreen','grayish blue-green',[94,113,106],'5e716a',6189418],
['nbs-iscc-r:grayishblueviolet.196','grayishblueviolet','grayish blue-violet ',[84,90,167],'545aa7',5528231],
['nbs-iscc-r:grayishblueviolet.200','grayishblueviolet','grayish blue-violet ',[78,81,128],'4e5180',5132672],
['nbs-iscc-r:grayishblueviolet.207','grayishblueviolet','grayish blue-violet ',[96,78,151],'604e97',6311575],
['nbs-iscc-r:grayishblueviolet.211','grayishblueviolet','grayish blue-violet ',[96,78,129],'604e81',6311553],
['nbs-iscc-r:grayishblueviolet.199','grayishblueviolet','grayish blue-violet ',[135,145,191],'8791bf',8884671],
['nbs-iscc-r:grayishlavender.227','grayishlavender','grayish lavender',[170,152,169],'aa98a9',11180201],
['nbs-iscc-r:grayisholive.109','grayisholive','grayish olive',[140,135,103],'8c8767',9209703],
['nbs-iscc-r:grayishviolaceousblue.199','grayishviolaceousblue','grayish violaceous blue',[135,145,191],'8791bf',8884671],
['nbs-iscc-r:grayishvioletblue.210','grayishvioletblue','grayish violet-blue',[140,130,182],'8c82b6',9208502],
['nbs-iscc-r:grayishvioletblue.211','grayishvioletblue','grayish violet-blue',[96,78,129],'604e81',6311553],
['nbs-iscc-r:greenblueslate.186','greenblueslate','green-blue slate',[83,104,120],'536878',5466232],
['nbs-iscc-r:greenishglaucous.148','greenishglaucous','greenish glaucous',[199,230,215],'c7e6d7',13100759],
['nbs-iscc-r:greenishglaucousblue.163','greenishglaucousblue','greenish glaucous-blue',[102,173,164],'66ada4',6729124],
['nbs-iscc-r:greenishslateblack.151','greenishslateblack','greenish slate-black',[58,75,71],'3a4b47',3820359],
['nbs-iscc-r:greenishslateblack.156','greenishslateblack','greenish slate-black',[78,87,85],'4e5755',5134165],
['nbs-iscc-r:greenishslateblack.266','greenishslateblack','greenish slate-black',[85,85,85],'555555',5592405],
['nbs-iscc-r:greenishslateblack.267','greenishslateblack','greenish slate-black',[34,34,34],'222222',2236962],
['nbs-iscc-r:greenishyellow.99','greenishyellow','greenish yellow',[190,183,46],'beb72e',12498734],
['nbs-iscc-r:greenyellow.98','greenyellow','green-yellow',[233,228,80],'e9e450',15328336],
['nbs-iscc-r:grenadine.35','grenadine','grenadine',[217,96,59],'d9603b',14245947],
['nbs-iscc-r:grenadinepink.26','grenadinepink','grenadine pink',[248,131,121],'f88379',16286585],
['nbs-iscc-r:grenadinered.34','grenadinered','grenadine red ',[226,88,34],'e25822',14833698],
['nbs-iscc-r:guineagreen.160','guineagreen','guinea green',[0,122,116],'007a74',31348],
['nbs-iscc-r:gullgray.265','gullgray','gull gray',[132,132,130],'848482',8684674],
['nbs-iscc-r:haematitered.46','haematitered','haematite red',[103,76,71],'674c47',6769735],
['nbs-iscc-r:haematoxylinviolet.223','haematoxylinviolet','haematoxylin violet',[134,96,142],'86608e',8806542],
['nbs-iscc-r:hairbrown.64','hairbrown','hair brown',[91,80,79],'5b504f',5984335],
['nbs-iscc-r:hathigray.154','hathigray','hathi gray',[178,190,181],'b2beb5',11714229],
['nbs-iscc-r:hathigray.155','hathigray','hathi gray',[125,137,132],'7d8984',8227204],
['nbs-iscc-r:hathigray.264','hathigray','hathi gray',[185,184,181],'b9b8b5',12171445],
['nbs-iscc-r:hathigray.265','hathigray','hathi gray',[132,132,130],'848482',8684674],
['nbs-iscc-r:haysblue.196','haysblue','hay\'s blue',[84,90,167],'545aa7',5528231],
['nbs-iscc-r:haysbrown.19','haysbrown','hay\'s brown',[144,93,93],'905d5d',9461085],
['nbs-iscc-r:haysbrown.46','haysbrown','hay\'s brown',[103,76,71],'674c47',6769735],
['nbs-iscc-r:haysgreen.136','haysgreen','hay\'s green',[103,146,103],'679267',6787687],
['nbs-iscc-r:haysgreen.137','haysgreen','hay\'s green',[53,94,59],'355e3b',3497531],
['nbs-iscc-r:hayslilac.222','hayslilac','hay\'s lilac',[182,149,192],'b695c0',11965888],
['nbs-iscc-r:haysmaroon.20','haysmaroon','hay\'s maroon',[84,61,63],'543d3f',5520703],
['nbs-iscc-r:haysrusset.43','haysrusset','hay\'s russet',[121,68,59],'79443b',7947323],
['nbs-iscc-r:hazel.55','hazel','hazel',[128,70,27],'80461b',8406555],
['nbs-iscc-r:heliotropegray.227','heliotropegray','heliotrope gray',[170,152,169],'aa98a9',11180201],
['nbs-iscc-r:heliotropegray.228','heliotropegray','heliotrope gray',[121,104,120],'796878',7956600],
['nbs-iscc-r:heliotropeslate.228','heliotropeslate','heliotrope slate',[121,104,120],'796878',7956600],
['nbs-iscc-r:helleborered.258','helleborered','hellebore red',[168,81,110],'a8516e',11030894],
['nbs-iscc-r:helvetiablue.196','helvetiablue','helvetia blue',[84,90,167],'545aa7',5528231],
['nbs-iscc-r:hermosapink.2','hermosapink','hermosa pink',[234,147,153],'ea9399',15373209],
['nbs-iscc-r:hessianbrown.46','hessianbrown','hessian brown',[103,76,71],'674c47',6769735],
['nbs-iscc-r:hessianbrown.47','hessianbrown','hessian brown',[67,48,46],'43302e',4403246],
['nbs-iscc-r:honeyyellow.87','honeyyellow','honey yellow',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-r:honeyyellow.88','honeyyellow','honey yellow',[171,145,68],'ab9144',11243844],
['nbs-iscc-r:hortenseblue.182','hortenseblue','hortense blue',[67,107,149],'436b95',4418453],
['nbs-iscc-r:hortenseviolet.217','hortenseviolet','hortense violet',[211,153,230],'d399e6',13867494],
['nbs-iscc-r:hyacinthblue.197','hyacinthblue','hyacinth blue',[39,36,88],'272458',2565208],
['nbs-iscc-r:hyacinthblue.211','hyacinthblue','hyacinth blue',[96,78,129],'604e81',6311553],
['nbs-iscc-r:hyacinthviolet.219','hyacinthviolet','hyacinth violet',[96,47,107],'602f6b',6303595],
['nbs-iscc-r:hydrangeapink.28','hydrangeapink','hydrangea pink',[244,194,194],'f4c2c2',16040642],
['nbs-iscc-r:hydrangeapink.29','hydrangeapink','hydrangea pink',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-r:hydrangeared.19','hydrangeared','hydrangea red',[144,93,93],'905d5d',9461085],
['nbs-iscc-r:hyssopviolet.223','hyssopviolet','hyssop violet',[134,96,142],'86608e',8806542],
['nbs-iscc-r:indianlake.258','indianlake','indian lake',[168,81,110],'a8516e',11030894],
['nbs-iscc-r:indianlake.262','indianlake','indian lake',[145,95,109],'915f6d',9527149],
['nbs-iscc-r:indianpurple.229','indianpurple','indian purple',[80,64,77],'50404d',5259341],
['nbs-iscc-r:indianpurple.259','indianpurple','indian purple',[103,49,71],'673147',6762823],
['nbs-iscc-r:indianred.19','indianred','indian red',[144,93,93],'905d5d',9461085],
['nbs-iscc-r:indigoblue.183','indigoblue','indigo blue',[0,48,78],'00304e',12366],
['nbs-iscc-r:indigoblue.186','indigoblue','indigo blue',[83,104,120],'536878',5466232],
['nbs-iscc-r:indigoblue.187','indigoblue','indigo blue',[54,69,79],'36454f',3556687],
['nbs-iscc-r:indigoblue.204','indigoblue','indigo blue',[76,81,109],'4c516d',5001581],
['nbs-iscc-r:indulinblue.204','indulinblue','indulin blue',[76,81,109],'4c516d',5001581],
['nbs-iscc-r:invisiblegreen.151','invisiblegreen','invisible green',[58,75,71],'3a4b47',3820359],
['nbs-iscc-r:invisiblegreen.165','invisiblegreen','invisible green',[0,75,73],'004b49',19273],
['nbs-iscc-r:irongray.113','irongray','iron gray',[87,85,76],'57554c',5723468],
['nbs-iscc-r:isabellacolor.91','isabellacolor','isabella color',[161,143,96],'a18f60',10588000],
['nbs-iscc-r:italianblue.169','italianblue','italian blue',[0,119,145],'007791',30609],
['nbs-iscc-r:ivoryyellow.89','ivoryyellow','ivory yellow',[243,229,171],'f3e5ab',15984043],
['nbs-iscc-r:ivygreen.107','ivygreen','ivy green',[102,93,30],'665d1e',6708510],
['nbs-iscc-r:ivygreen.110','ivygreen','ivy green',[91,88,66],'5b5842',5986370],
['nbs-iscc-r:jadegreen.125','jadegreen','jade green',[74,93,35],'4a5d23',4873507],
['nbs-iscc-r:jadegreen.127','jadegreen','jade green',[81,87,68],'515744',5330756],
['nbs-iscc-r:japanrose.29','japanrose','japan rose',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-r:jaspergreen.145','jaspergreen','jasper green',[59,120,97],'3b7861',3897441],
['nbs-iscc-r:jasperpink.3','jasperpink','jasper pink',[228,113,122],'e4717a',14971258],
['nbs-iscc-r:jasperpink.26','jasperpink','jasper pink',[248,131,121],'f88379',16286585],
['nbs-iscc-r:jasperpink.27','jasperpink','jasper pink',[230,103,97],'e66761',15099745],
['nbs-iscc-r:jasperred.15','jasperred','jasper red',[171,78,82],'ab4e52',11226706],
['nbs-iscc-r:javelgreen.102','javelgreen','javel green',[185,180,89],'b9b459',12170329],
['nbs-iscc-r:jayblue.182','jayblue','jay blue ',[67,107,149],'436b95',4418453],
['nbs-iscc-r:jayblue.199','jayblue','jay blue ',[135,145,191],'8791bf',8884671],
['nbs-iscc-r:jouvenceblue.164','jouvenceblue','jouvence blue',[49,120,115],'317873',3242099],
['nbs-iscc-r:jouvenceblue.173','jouvenceblue','jouvence blue',[54,117,136],'367588',3569032],
['nbs-iscc-r:kaiserbrown.55','kaiserbrown','kaiser brown',[128,70,27],'80461b',8406555],
['nbs-iscc-r:kildaregreen.120','kildaregreen','kildare green',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-r:killarneygreen.136','killarneygreen','killarney green ',[103,146,103],'679267',6787687],
['nbs-iscc-r:kingsblue.181','kingsblue','king\'s blue',[112,163,204],'70a3cc',7381964],
['nbs-iscc-r:kronbergsgreen.106','kronbergsgreen','kronberg\'s green',[134,126,54],'867e36',8814134],
['nbs-iscc-r:kronbergsgreen.107','kronbergsgreen','kronberg\'s green',[102,93,30],'665d1e',6708510],
['nbs-iscc-r:laeliapink.240','laeliapink','laelia pink',[183,132,167],'b784a7',12027047],
['nbs-iscc-r:laeliapink.248','laeliapink','laelia pink',[222,111,161],'de6fa1',14577569],
['nbs-iscc-r:lafrancepink.1','lafrancepink','la france pink',[255,181,186],'ffb5ba',16758202],
['nbs-iscc-r:lavender.209','lavender','lavender',[220,208,255],'dcd0ff',14471423],
['nbs-iscc-r:lavender.213','lavender','lavender',[196,195,221],'c4c3dd',12895197],
['nbs-iscc-r:lavenderblue.195','lavenderblue','lavender-blue',[108,121,184],'6c79b8',7109048],
['nbs-iscc-r:lavenderblue.199','lavenderblue','lavender-blue',[135,145,191],'8791bf',8884671],
['nbs-iscc-r:lavendergray.202','lavendergray','lavender-gray',[192,200,225],'c0c8e1',12634337],
['nbs-iscc-r:lavendergray.203','lavendergray','lavender-gray',[140,146,172],'8c92ac',9212588],
['nbs-iscc-r:lavendergray.213','lavendergray','lavender-gray',[196,195,221],'c4c3dd',12895197],
['nbs-iscc-r:lavendergray.214','lavendergray','lavender-gray',[150,144,171],'9690ab',9867435],
['nbs-iscc-r:lavenderviolet.206','lavenderviolet','lavender-violet',[126,115,184],'7e73b8',8287160],
['nbs-iscc-r:leafgreen.127','leafgreen','leaf green',[81,87,68],'515744',5330756],
['nbs-iscc-r:leitchsblue.178','leitchsblue','leitch\'s blue',[0,103,165],'0067a5',26533],
['nbs-iscc-r:lemonchrome.84','lemonchrome','lemon chrome',[212,175,55],'d4af37',13938487],
['nbs-iscc-r:lemonyellow.98','lemonyellow','lemon yellow',[233,228,80],'e9e450',15328336],
['nbs-iscc-r:lemonyellow.99','lemonyellow','lemon yellow',[190,183,46],'beb72e',12498734],
['nbs-iscc-r:lettucegreen.120','lettucegreen','lettuce green',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-r:lichengreen.148','lichengreen','lichen green',[199,230,215],'c7e6d7',13100759],
['nbs-iscc-r:lightaliceblue.181','lightaliceblue','light alice blue',[112,163,204],'70a3cc',7381964],
['nbs-iscc-r:lightaliceblue.185','lightaliceblue','light alice blue',[145,163,176],'91a3b0',9544624],
['nbs-iscc-r:lightamparoblue.177','lightamparoblue','light amparo blue',[73,151,208],'4997d0',4822992],
['nbs-iscc-r:lightamparopurple.217','lightamparopurple','light amparo purple',[211,153,230],'d399e6',13867494],
['nbs-iscc-r:lightbicegreen.120','lightbicegreen','light bice green',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-r:lightbluegreen.140','lightbluegreen','light blue-green',[62,180,137],'3eb489',4109449],
['nbs-iscc-r:lightbluegreen.144','lightbluegreen','light blue-green',[106,171,142],'6aab8e',6990734],
['nbs-iscc-r:lightblueviolet.194','lightblueviolet','light blue-violet',[48,38,122],'30267a',3155578],
['nbs-iscc-r:lightbluishviolet.195','lightbluishviolet','light bluish violet',[108,121,184],'6c79b8',7109048],
['nbs-iscc-r:lightbluishviolet.196','lightbluishviolet','light bluish violet',[84,90,167],'545aa7',5528231],
['nbs-iscc-r:lightbluishviolet.206','lightbluishviolet','light bluish violet',[126,115,184],'7e73b8',8287160],
['nbs-iscc-r:lightbrownishdrab.60','lightbrownishdrab','light brownish drab',[149,128,112],'958070',9797744],
['nbs-iscc-r:lightbrownisholive.94','lightbrownisholive','light brownish olive',[150,113,23],'967117',9859351],
['nbs-iscc-r:lightbrownishvinaceous.8','lightbrownishvinaceous','light brownish vinaceous',[196,174,173],'c4aead',12889773],
['nbs-iscc-r:lightbuff.73','lightbuff','light buff',[250,214,165],'fad6a5',16438949],
['nbs-iscc-r:lightbuff.89','lightbuff','light buff',[243,229,171],'f3e5ab',15984043],
['nbs-iscc-r:lightcadetblue.177','lightcadetblue','light cadet blue',[73,151,208],'4997d0',4822992],
['nbs-iscc-r:lightcadmium.82','lightcadmium','light cadmium',[243,195,0],'f3c300',15975168],
['nbs-iscc-r:lightcampanulablue.195','lightcampanulablue','light campanula blue',[108,121,184],'6c79b8',7109048],
['nbs-iscc-r:lightcelandinegreen.154','lightcelandinegreen','light celandine green',[178,190,181],'b2beb5',11714229],
['nbs-iscc-r:lightcelandinegreen.155','lightcelandinegreen','light celandine green',[125,137,132],'7d8984',8227204],
['nbs-iscc-r:lightcendregreen.130','lightcendregreen','light cendre green',[131,211,125],'83d37d',8639357],
['nbs-iscc-r:lightcendregreen.135','lightcendregreen','light cendre green',[147,197,146],'93c592',9684370],
['nbs-iscc-r:lightceruleanblue.181','lightceruleanblue','light cerulean blue',[112,163,204],'70a3cc',7381964],
['nbs-iscc-r:lightchalcedonyyellow.101','lightchalcedonyyellow','light chalcedony yellow',[234,230,121],'eae679',15394425],
['nbs-iscc-r:lightchicoryblue.210','lightchicoryblue','light chicory blue',[140,130,182],'8c82b6',9208502],
['nbs-iscc-r:lightcinnamondrab.45','lightcinnamondrab','light cinnamon-drab',[151,127,115],'977f73',9928563],
['nbs-iscc-r:lightcinnamondrab.60','lightcinnamondrab','light cinnamon-drab',[149,128,112],'958070',9797744],
['nbs-iscc-r:lightcolumbiablue.181','lightcolumbiablue','light columbia blue',[112,163,204],'70a3cc',7381964],
['nbs-iscc-r:lightcongopink.28','lightcongopink','light congo pink',[244,194,194],'f4c2c2',16040642],
['nbs-iscc-r:lightcongopink.29','lightcongopink','light congo pink',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-r:lightcoralred.37','lightcoralred','light coral red',[203,109,81],'cb6d51',13331793],
['nbs-iscc-r:lightcorinthianred.6','lightcorinthianred','light corinthian red',[192,128,129],'c08081',12615809],
['nbs-iscc-r:lightcressgreen.120','lightcressgreen','light cress green',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-r:lightdanubegreen.145','lightdanubegreen','light danube green',[59,120,97],'3b7861',3897441],
['nbs-iscc-r:lightdrab.79','lightdrab','light drab',[174,155,130],'ae9b82',11443074],
['nbs-iscc-r:lightdullbluishviolet.210','lightdullbluishviolet','light dull bluish violet',[140,130,182],'8c82b6',9208502],
['nbs-iscc-r:lightdullglaucousblue.172','lightdullglaucousblue','light dull glaucous-blue',[102,170,188],'66aabc',6728380],
['nbs-iscc-r:lightdullglaucousblue.185','lightdullglaucousblue','light dull glaucous-blue',[145,163,176],'91a3b0',9544624],
['nbs-iscc-r:lightdullgreenyellow.119','lightdullgreenyellow','light dull green-yellow',[201,220,137],'c9dc89',13229193],
['nbs-iscc-r:lightelmgreen.120','lightelmgreen','light elm green',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-r:lightfluoritegreen.119','lightfluoritegreen','light fluorite green',[201,220,137],'c9dc89',13229193],
['nbs-iscc-r:lightforgetmenotblue.181','lightforgetmenotblue','light forget-me-not blue ',[112,163,204],'70a3cc',7381964],
['nbs-iscc-r:lightglaucousblue.171','lightglaucousblue','light glaucous-blue',[156,209,220],'9cd1dc',10277340],
['nbs-iscc-r:lightgrapegreen.120','lightgrapegreen','light grape green',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-r:lightgrayishblueviolet.181','lightgrayishblueviolet','light grayish blue-violet',[112,163,204],'70a3cc',7381964],
['nbs-iscc-r:lightgrayishblueviolet.199','lightgrayishblueviolet','light grayish blue-violet',[135,145,191],'8791bf',8884671],
['nbs-iscc-r:lightgrayisholive.109','lightgrayisholive','light grayish olive',[140,135,103],'8c8767',9209703],
['nbs-iscc-r:lightgrayisholive.112','lightgrayisholive','light grayish olive',[138,135,118],'8a8776',9078646],
['nbs-iscc-r:lightgrayishvinaceous.32','lightgrayishvinaceous','light grayish vinaceous',[199,173,163],'c7ada3',13086115],
['nbs-iscc-r:lightgrayishvioletblue.195','lightgrayishvioletblue','light grayish violet-blue',[108,121,184],'6c79b8',7109048],
['nbs-iscc-r:lightgreenishyellow.98','lightgreenishyellow','light greenish yellow',[233,228,80],'e9e450',15328336],
['nbs-iscc-r:lightgreenishyellow.99','lightgreenishyellow','light greenish yellow',[190,183,46],'beb72e',12498734],
['nbs-iscc-r:lightgreenyellow.101','lightgreenyellow','light green-yellow',[234,230,121],'eae679',15394425],
['nbs-iscc-r:lightgullgray.264','lightgullgray','light gull gray',[185,184,181],'b9b8b5',12171445],
['nbs-iscc-r:lightheliotropegray.227','lightheliotropegray','light heliotrope gray',[170,152,169],'aa98a9',11180201],
['nbs-iscc-r:lighthelleboregreen.120','lighthelleboregreen','light hellebore green',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-r:lighthelleboregreen.122','lighthelleboregreen','light hellebore green',[143,151,121],'8f9779',9410425],
['nbs-iscc-r:lighthortenseviolet.217','lighthortenseviolet','light hortense violet',[211,153,230],'d399e6',13867494],
['nbs-iscc-r:lighthyssopviolet.210','lighthyssopviolet','light hyssop violet',[140,130,182],'8c82b6',9208502],
['nbs-iscc-r:lightjasperred.3','lightjasperred','light jasper red',[228,113,122],'e4717a',14971258],
['nbs-iscc-r:lightjasperred.15','lightjasperred','light jasper red',[171,78,82],'ab4e52',11226706],
['nbs-iscc-r:lightkingsblue.184','lightkingsblue','light king\'s blue',[188,212,230],'bcd4e6',12375270],
['nbs-iscc-r:lightkingsblue.185','lightkingsblue','light king\'s blue',[145,163,176],'91a3b0',9544624],
['nbs-iscc-r:lightlavenderblue.199','lightlavenderblue','light lavender-blue',[135,145,191],'8791bf',8884671],
['nbs-iscc-r:lightlavenderviolet.210','lightlavenderviolet','light lavender-violet',[140,130,182],'8c82b6',9208502],
['nbs-iscc-r:lightlobeliaviolet.222','lightlobeliaviolet','light lobelia violet',[182,149,192],'b695c0',11965888],
['nbs-iscc-r:lightlumieregreen.119','lightlumieregreen','light lumiere green',[201,220,137],'c9dc89',13229193],
['nbs-iscc-r:lightmallowpurple.248','lightmallowpurple','light mallow purple',[222,111,161],'de6fa1',14577569],
['nbs-iscc-r:lightmauve.210','lightmauve','light mauve',[140,130,182],'8c82b6',9208502],
['nbs-iscc-r:lightmediciblue.186','lightmediciblue','light medici blue',[83,104,120],'536878',5466232],
['nbs-iscc-r:lightmethylblue.177','lightmethylblue','light methyl blue ',[73,151,208],'4997d0',4822992],
['nbs-iscc-r:lightmethylblue.178','lightmethylblue','light methyl blue ',[0,103,165],'0067a5',26533],
['nbs-iscc-r:lightmineralgray.154','lightmineralgray','light mineral gray ',[178,190,181],'b2beb5',11714229],
['nbs-iscc-r:lightmousegray.63','lightmousegray','light mouse gray ',[142,130,121],'8e8279',9339513],
['nbs-iscc-r:lightneropalinblue.185','lightneropalinblue','light neropalin blue',[145,163,176],'91a3b0',9544624],
['nbs-iscc-r:lightneutralgray.265','lightneutralgray','light neutral gray',[132,132,130],'848482',8684674],
['nbs-iscc-r:lightniagaragreen.163','lightniagaragreen','light niagara green',[102,173,164],'66ada4',6729124],
['nbs-iscc-r:lightochraceousbuff.71','lightochraceousbuff','light ochraceous-buff',[227,168,87],'e3a857',14919767],
['nbs-iscc-r:lightochraceoussalmon.73','lightochraceoussalmon','light ochraceous-salmon',[250,214,165],'fad6a5',16438949],
['nbs-iscc-r:lightolivegray.93','lightolivegray','light olive-gray',[191,184,165],'bfb8a5',12564645],
['nbs-iscc-r:lightorangeyellow.71','lightorangeyellow','light orange-yellow',[227,168,87],'e3a857',14919767],
['nbs-iscc-r:lightorangeyellow.84','lightorangeyellow','light orange-yellow',[212,175,55],'d4af37',13938487],
['nbs-iscc-r:lightorientalgreen.135','lightorientalgreen','light oriental green',[147,197,146],'93c592',9684370],
['nbs-iscc-r:lightparisgreen.135','lightparisgreen','light paris green',[147,197,146],'93c592',9684370],
['nbs-iscc-r:lightpaynesgray.185','lightpaynesgray','light payne\'s gray',[145,163,176],'91a3b0',9544624],
['nbs-iscc-r:lightpaynesgray.191','lightpaynesgray','light payne\'s gray',[129,135,139],'81878b',8488843],
['nbs-iscc-r:lightperillapurple.241','lightperillapurple','light perilla purple',[145,92,131],'915c83',9526403],
['nbs-iscc-r:lightperillapurple.245','lightperillapurple','light perilla purple',[131,100,121],'836479',8610937],
['nbs-iscc-r:lightphloxpurple.248','lightphloxpurple','light phlox purple',[222,111,161],'de6fa1',14577569],
['nbs-iscc-r:lightpinkishcinnamon.28','lightpinkishcinnamon','light pinkish cinnamon',[244,194,194],'f4c2c2',16040642],
['nbs-iscc-r:lightpinkishcinnamon.29','lightpinkishcinnamon','light pinkish cinnamon',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-r:lightpinkishlilac.249','lightpinkishlilac','light pinkish lilac',[239,187,204],'efbbcc',15711180],
['nbs-iscc-r:lightplumbagogray.233','lightplumbagogray','light plumbago gray',[139,133,137],'8b8589',9143689],
['nbs-iscc-r:lightporcelaingreen.163','lightporcelaingreen','light porcelain green',[102,173,164],'66ada4',6729124],
['nbs-iscc-r:lightporcelaingreen.164','lightporcelaingreen','light porcelain green',[49,120,115],'317873',3242099],
['nbs-iscc-r:lightpurpledrab.228','lightpurpledrab','light purple-drab',[121,104,120],'796878',7956600],
['nbs-iscc-r:lightpurplishgray.233','lightpurplishgray','light purplish gray',[139,133,137],'8b8589',9143689],
['nbs-iscc-r:lightpurplishgray.265','lightpurplishgray','light purplish gray',[132,132,130],'848482',8684674],
['nbs-iscc-r:lightpurplishvinaceous.5','lightpurplishvinaceous','light purplish vinaceous',[222,165,164],'dea5a4',14591396],
['nbs-iscc-r:lightquakerdrab.233','lightquakerdrab','light quaker drab ',[139,133,137],'8b8589',9143689],
['nbs-iscc-r:lightrosolanepurple.248','lightrosolanepurple','light rosolane purple',[222,111,161],'de6fa1',14577569],
['nbs-iscc-r:lightrussetvinaceous.42','lightrussetvinaceous','light russet-vinaceous',[168,124,109],'a87c6d',11041901],
['nbs-iscc-r:lightsalmonorange.53','lightsalmonorange','light salmon-orange',[217,144,88],'d99058',14258264],
['nbs-iscc-r:lightsealbrown.23','lightsealbrown','light seal brown',[92,80,79],'5c504f',6049871],
['nbs-iscc-r:lightskyblue.185','lightskyblue','light sky blue',[145,163,176],'91a3b0',9544624],
['nbs-iscc-r:lightsoftblueviolet.195','lightsoftblueviolet','light soft blue-violet',[108,121,184],'6c79b8',7109048],
['nbs-iscc-r:lightsoftblueviolet.206','lightsoftblueviolet','light soft blue-violet',[126,115,184],'7e73b8',8287160],
['nbs-iscc-r:lightsquillblue.181','lightsquillblue','light squill blue',[112,163,204],'70a3cc',7381964],
['nbs-iscc-r:lightsulphategreen.163','lightsulphategreen','light sulphate green',[102,173,164],'66ada4',6729124],
['nbs-iscc-r:lightterreverte.164','lightterreverte','light terre verte',[49,120,115],'317873',3242099],
['nbs-iscc-r:lightturtlegreen.119','lightturtlegreen','light turtle green',[201,220,137],'c9dc89',13229193],
['nbs-iscc-r:lighttyrianblue.186','lighttyrianblue','light tyrian blue',[83,104,120],'536878',5466232],
['nbs-iscc-r:lighttyrianblue.203','lighttyrianblue','light tyrian blue',[140,146,172],'8c92ac',9212588],
['nbs-iscc-r:lighttyrianblue.204','lighttyrianblue','light tyrian blue',[76,81,109],'4c516d',5001581],
['nbs-iscc-r:lightvarleysgray.227','lightvarleysgray','light varley\'s gray',[170,152,169],'aa98a9',11180201],
['nbs-iscc-r:lightvinaceouscinnamon.73','lightvinaceouscinnamon','light vinaceous-cinnamon',[250,214,165],'fad6a5',16438949],
['nbs-iscc-r:lightvinaceousdrab.19','lightvinaceousdrab','light vinaceous-drab',[144,93,93],'905d5d',9461085],
['nbs-iscc-r:lightvinaceousfawn.32','lightvinaceousfawn','light vinaceous-fawn',[199,173,163],'c7ada3',13086115],
['nbs-iscc-r:lightvinaceousfawn.33','lightvinaceousfawn','light vinaceous-fawn',[194,172,153],'c2ac99',12758169],
['nbs-iscc-r:lightvinaceousgray.227','lightvinaceousgray','light vinaceous-gray',[170,152,169],'aa98a9',11180201],
['nbs-iscc-r:lightvinaceousgray.232','lightvinaceousgray','light vinaceous-gray',[191,185,189],'bfb9bd',12564925],
['nbs-iscc-r:lightvinaceousgray.233','lightvinaceousgray','light vinaceous-gray',[139,133,137],'8b8589',9143689],
['nbs-iscc-r:lightvinaceousgray.253','lightvinaceousgray','light vinaceous-gray',[195,166,177],'c3a6b1',12822193],
['nbs-iscc-r:lightvinaceouslilac.240','lightvinaceouslilac','light vinaceous-lilac',[183,132,167],'b784a7',12027047],
['nbs-iscc-r:lightvinaceouslilac.244','lightvinaceouslilac','light vinaceous-lilac',[170,138,158],'aa8a9e',11176606],
['nbs-iscc-r:lightvinaceouslilac.250','lightvinaceouslilac','light vinaceous-lilac',[213,151,174],'d597ae',13997998],
['nbs-iscc-r:lightvinaceouslilac.253','lightvinaceouslilac','light vinaceous-lilac',[195,166,177],'c3a6b1',12822193],
['nbs-iscc-r:lightvinaceouspurple.245','lightvinaceouspurple','light vinaceous-purple',[131,100,121],'836479',8610937],
['nbs-iscc-r:lightviolet.195','lightviolet','light violet',[108,121,184],'6c79b8',7109048],
['nbs-iscc-r:lightviolet.206','lightviolet','light violet',[126,115,184],'7e73b8',8287160],
['nbs-iscc-r:lightvioletblue.196','lightvioletblue','light violet-blue',[84,90,167],'545aa7',5528231],
['nbs-iscc-r:lightvioletgray.233','lightvioletgray','light violet-gray',[139,133,137],'8b8589',9143689],
['nbs-iscc-r:lightvioletgray.265','lightvioletgray','light violet-gray',[132,132,130],'848482',8684674],
['nbs-iscc-r:lightvioletplumbeous.185','lightvioletplumbeous','light violet-plumbeous ',[145,163,176],'91a3b0',9544624],
['nbs-iscc-r:lightvioletplumbeous.186','lightvioletplumbeous','light violet-plumbeous ',[83,104,120],'536878',5466232],
['nbs-iscc-r:lightvioletplumbeous.203','lightvioletplumbeous','light violet-plumbeous ',[140,146,172],'8c92ac',9212588],
['nbs-iscc-r:lightviridinegreen.119','lightviridinegreen','light viridine green',[201,220,137],'c9dc89',13229193],
['nbs-iscc-r:lightviridineyellow.101','lightviridineyellow','light viridine yellow',[234,230,121],'eae679',15394425],
['nbs-iscc-r:lightwindsorblue.199','lightwindsorblue','light windsor blue',[135,145,191],'8791bf',8884671],
['nbs-iscc-r:lightwistariablue.199','lightwistariablue','light wistaria blue ',[135,145,191],'8791bf',8884671],
['nbs-iscc-r:lightwistariablue.210','lightwistariablue','light wistaria blue ',[140,130,182],'8c82b6',9208502],
['nbs-iscc-r:lightwistariaviolet.210','lightwistariaviolet','light wistaria violet',[140,130,182],'8c82b6',9208502],
['nbs-iscc-r:lightyellowgreen.116','lightyellowgreen','light yellow-green',[189,218,87],'bdda57',12442199],
['nbs-iscc-r:lightyellowgreen.119','lightyellowgreen','light yellow-green',[201,220,137],'c9dc89',13229193],
['nbs-iscc-r:lightyellowisholive.103','lightyellowisholive','light yellowish olive',[152,148,62],'98943e',9999422],
['nbs-iscc-r:lilac.222','lilac','lilac',[182,149,192],'b695c0',11965888],
['nbs-iscc-r:lilac.227','lilac','lilac',[170,152,169],'aa98a9',11180201],
['nbs-iscc-r:lilacgray.232','lilacgray','lilac gray',[191,185,189],'bfb9bd',12564925],
['nbs-iscc-r:lilygreen.155','lilygreen','lily green',[125,137,132],'7d8984',8227204],
['nbs-iscc-r:limegreen.102','limegreen','lime green',[185,180,89],'b9b459',12170329],
['nbs-iscc-r:limegreen.103','limegreen','lime green',[152,148,62],'98943e',9999422],
['nbs-iscc-r:limegreen.106','limegreen','lime green',[134,126,54],'867e36',8814134],
['nbs-iscc-r:lincolngreen.127','lincolngreen','lincoln green',[81,87,68],'515744',5330756],
['nbs-iscc-r:liseranpurple.237','liseranpurple','liseran purple',[158,79,136],'9e4f88',10375048],
['nbs-iscc-r:liseranpurple.240','liseranpurple','liseran purple',[183,132,167],'b784a7',12027047],
['nbs-iscc-r:liseranpurple.241','liseranpurple','liseran purple',[145,92,131],'915c83',9526403],
['nbs-iscc-r:liseranpurple.248','liseranpurple','liseran purple',[222,111,161],'de6fa1',14577569],
['nbs-iscc-r:lithopurple.218','lithopurple','litho purple',[135,86,146],'875692',8869522],
['nbs-iscc-r:lithopurple.219','lithopurple','litho purple',[96,47,107],'602f6b',6303595],
['nbs-iscc-r:lithopurple.223','lithopurple','litho purple',[134,96,142],'86608e',8806542],
['nbs-iscc-r:liverbrown.43','liverbrown','liver brown',[121,68,59],'79443b',7947323],
['nbs-iscc-r:lividbrown.19','lividbrown','livid brown',[144,93,93],'905d5d',9461085],
['nbs-iscc-r:lividpink.4','lividpink','livid pink',[249,204,202],'f9ccca',16370890],
['nbs-iscc-r:lividpink.5','lividpink','livid pink',[222,165,164],'dea5a4',14591396],
['nbs-iscc-r:lividpink.7','lividpink','livid pink',[234,216,215],'ead8d7',15390935],
['nbs-iscc-r:lividpink.8','lividpink','livid pink',[196,174,173],'c4aead',12889773],
['nbs-iscc-r:lividpurple.241','lividpurple','livid purple',[145,92,131],'915c83',9526403],
['nbs-iscc-r:lividviolet.223','lividviolet','livid violet',[134,96,142],'86608e',8806542],
['nbs-iscc-r:lividviolet.228','lividviolet','livid violet',[121,104,120],'796878',7956600],
['nbs-iscc-r:lobeliaviolet.222','lobeliaviolet','lobelia violet',[182,149,192],'b695c0',11965888],
['nbs-iscc-r:lumiereblue.163','lumiereblue','lumiere blue',[102,173,164],'66ada4',6729124],
['nbs-iscc-r:lumiereblue.172','lumiereblue','lumiere blue',[102,170,188],'66aabc',6728380],
['nbs-iscc-r:lumieregreen.119','lumieregreen','lumiere green',[201,220,137],'c9dc89',13229193],
['nbs-iscc-r:lumieregreen.120','lumieregreen','lumiere green',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-r:lyonsblue.178','lyonsblue','lyons blue',[0,103,165],'0067a5',26533],
['nbs-iscc-r:madderblue.214','madderblue','madder blue',[150,144,171],'9690ab',9867435],
['nbs-iscc-r:madderblue.228','madderblue','madder blue',[121,104,120],'796878',7956600],
['nbs-iscc-r:madderbrown.19','madderbrown','madder brown',[144,93,93],'905d5d',9461085],
['nbs-iscc-r:madderviolet.224','madderviolet','madder violet',[86,60,92],'563c5c',5651548],
['nbs-iscc-r:magenta.237','magenta','magenta',[158,79,136],'9e4f88',10375048],
['nbs-iscc-r:mahoganyred.43','mahoganyred','mahogany red',[121,68,59],'79443b',7947323],
['nbs-iscc-r:maizeyellow.86','maizeyellow','maize yellow',[248,222,126],'f8de7e',16309886],
['nbs-iscc-r:malachitegreen.135','malachitegreen','malachite green',[147,197,146],'93c592',9684370],
['nbs-iscc-r:mallowpink.247','mallowpink','mallow pink',[230,143,172],'e68fac',15110060],
['nbs-iscc-r:mallowpurple.236','mallowpurple','mallow purple',[135,0,116],'870074',8847476],
['nbs-iscc-r:mallowpurple.248','mallowpurple','mallow purple',[222,111,161],'de6fa1',14577569],
['nbs-iscc-r:manganeseviolet.218','manganeseviolet','manganese violet',[135,86,146],'875692',8869522],
['nbs-iscc-r:margueriteyellow.89','margueriteyellow','marguerite yellow',[243,229,171],'f3e5ab',15984043],
['nbs-iscc-r:marineblue.182','marineblue','marine blue',[67,107,149],'436b95',4418453],
['nbs-iscc-r:maroon.44','maroon','maroon',[62,29,30],'3e1d1e',4070686],
['nbs-iscc-r:marsbrown.58','marsbrown','mars brown',[111,78,55],'6f4e37',7294519],
['nbs-iscc-r:marsorange.37','marsorange','mars orange',[203,109,81],'cb6d51',13331793],
['nbs-iscc-r:marsorange.38','marsorange','mars orange',[158,71,50],'9e4732',10372914],
['nbs-iscc-r:marsviolet.229','marsviolet','mars violet',[80,64,77],'50404d',5259341],
['nbs-iscc-r:marsyellow.69','marsyellow','mars yellow',[201,133,0],'c98500',13206784],
['nbs-iscc-r:marsyellow.72','marsyellow','mars yellow',[190,138,61],'be8a3d',12487229],
['nbs-iscc-r:martiusyellow.101','martiusyellow','martius yellow',[234,230,121],'eae679',15394425],
['nbs-iscc-r:massicotyellow.86','massicotyellow','massicot yellow',[248,222,126],'f8de7e',16309886],
['nbs-iscc-r:massicotyellow.89','massicotyellow','massicot yellow',[243,229,171],'f3e5ab',15984043],
['nbs-iscc-r:mathewsblue.178','mathewsblue','mathew\'s blue',[0,103,165],'0067a5',26533],
['nbs-iscc-r:mathewspurple.218','mathewspurple','mathew\'s purple',[135,86,146],'875692',8869522],
['nbs-iscc-r:mathewspurple.223','mathewspurple','mathew\'s purple',[134,96,142],'86608e',8806542],
['nbs-iscc-r:mauve.206','mauve','mauve',[126,115,184],'7e73b8',8287160],
['nbs-iscc-r:mauvette.249','mauvette','mauvette',[239,187,204],'efbbcc',15711180],
['nbs-iscc-r:mazarineblue.177','mazarineblue','mazarine blue',[73,151,208],'4997d0',4822992],
['nbs-iscc-r:mazarineblue.181','mazarineblue','mazarine blue',[112,163,204],'70a3cc',7381964],
['nbs-iscc-r:meadowgreen.136','meadowgreen','meadow green',[103,146,103],'679267',6787687],
['nbs-iscc-r:medalbronze.95','medalbronze','medal bronze',[108,84,30],'6c541e',7099422],
['nbs-iscc-r:mediciblue.191','mediciblue','medici blue',[129,135,139],'81878b',8488843],
['nbs-iscc-r:methylblue.176','methylblue','methyl blue ',[0,161,194],'00a1c2',41410],
['nbs-iscc-r:methylgreen.160','methylgreen','methyl green ',[0,122,116],'007a74',31348],
['nbs-iscc-r:methylgreen.164','methylgreen','methyl green ',[49,120,115],'317873',3242099],
['nbs-iscc-r:microclinegreen.162','microclinegreen','microcline green ',[150,222,209],'96ded1',9887441],
['nbs-iscc-r:mignonettegreen.106','mignonettegreen','mignonette green ',[134,126,54],'867e36',8814134],
['nbs-iscc-r:mikadobrown.57','mikadobrown','mikado brown ',[166,123,91],'a67b5b',10910555],
['nbs-iscc-r:mikadoorange.50','mikadoorange','mikado orange ',[237,135,45],'ed872d',15566637],
['nbs-iscc-r:mineralgray.122','mineralgray','mineral gray ',[143,151,121],'8f9779',9410425],
['nbs-iscc-r:mineralgreen.117','mineralgreen','mineral green ',[126,159,46],'7e9f2e',8298286],
['nbs-iscc-r:mineralred.19','mineralred','mineral red ',[144,93,93],'905d5d',9461085],
['nbs-iscc-r:montpelliergreen.144','montpelliergreen','montpellier green ',[106,171,142],'6aab8e',6990734],
['nbs-iscc-r:montpelliergreen.145','montpelliergreen','montpellier green ',[59,120,97],'3b7861',3897441],
['nbs-iscc-r:moroccored.43','moroccored','morocco red ',[121,68,59],'79443b',7947323],
['nbs-iscc-r:motmotblue.173','motmotblue','motmot blue ',[54,117,136],'367588',3569032],
['nbs-iscc-r:motmotgreen.131','motmotgreen','motmot green ',[68,148,74],'44944a',4494410],
['nbs-iscc-r:motmotgreen.136','motmotgreen','motmot green ',[103,146,103],'679267',6787687],
['nbs-iscc-r:mousegray.63','mousegray','mouse gray ',[142,130,121],'8e8279',9339513],
['nbs-iscc-r:mulberrypurple.219','mulberrypurple','mulberry purple ',[96,47,107],'602f6b',6303595],
['nbs-iscc-r:mulberrypurple.224','mulberrypurple','mulberry purple ',[86,60,92],'563c5c',5651548],
['nbs-iscc-r:mummybrown.78','mummybrown','mummy brown ',[75,54,33],'4b3621',4929057],
['nbs-iscc-r:munichlake.11','munichlake','munich lake ',[190,0,50],'be0032',12451890],
['nbs-iscc-r:munichlake.12','munichlake','munich lake ',[188,63,74],'bc3f4a',12336970],
['nbs-iscc-r:mustardyellow.84','mustardyellow','mustard yellow ',[212,175,55],'d4af37',13938487],
['nbs-iscc-r:myrtlegreen.165','myrtlegreen','myrtle green ',[0,75,73],'004b49',19273],
['nbs-iscc-r:mythogreen.120','mythogreen','mytho green ',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-r:mythogreen.122','mythogreen','mytho green ',[143,151,121],'8f9779',9410425],
['nbs-iscc-r:naphthaleneviolet.242','naphthaleneviolet','naphthalene violet ',[93,57,84],'5d3954',6109524],
['nbs-iscc-r:naphthaleneyellow.101','naphthaleneyellow','naphthalene yellow ',[234,230,121],'eae679',15394425],
['nbs-iscc-r:naphthaleneyellow.104','naphthaleneyellow','naphthalene yellow ',[235,232,164],'ebe8a4',15460516],
['nbs-iscc-r:naplesyellow.86','naplesyellow','naples yellow ',[248,222,126],'f8de7e',16309886],
['nbs-iscc-r:natalbrown.61','natalbrown','natal brown',[99,81,71],'635147',6508871],
['nbs-iscc-r:navyblue.200','navyblue','navy blue',[78,81,128],'4e5180',5132672],
['nbs-iscc-r:navyblue.204','navyblue','navy blue',[76,81,109],'4c516d',5001581],
['nbs-iscc-r:neropalinblue.181','neropalinblue','neropalin blue',[112,163,204],'70a3cc',7381964],
['nbs-iscc-r:neutralgray.265','neutralgray','neutral gray',[132,132,130],'848482',8684674],
['nbs-iscc-r:neutralred.259','neutralred','neutral red',[103,49,71],'673147',6762823],
['nbs-iscc-r:neutralred.262','neutralred','neutral red',[145,95,109],'915f6d',9527149],
['nbs-iscc-r:neuvidergreen.135','neuvidergreen','neuvider green',[147,197,146],'93c592',9684370],
['nbs-iscc-r:nevagreen.116','nevagreen','neva green',[189,218,87],'bdda57',12442199],
['nbs-iscc-r:niagaragreen.149','niagaragreen','niagara green',[141,163,153],'8da399',9282457],
['nbs-iscc-r:niagaragreen.163','niagaragreen','niagara green',[102,173,164],'66ada4',6729124],
['nbs-iscc-r:nickelgreen.145','nickelgreen','nickel green',[59,120,97],'3b7861',3897441],
['nbs-iscc-r:nickelgreen.150','nickelgreen','nickel green',[94,113,106],'5e716a',6189418],
['nbs-iscc-r:nightgreen.115','nightgreen','night green',[141,182,0],'8db600',9287168],
['nbs-iscc-r:nigrosinblue.204','nigrosinblue','nigrosin blue',[76,81,109],'4c516d',5001581],
['nbs-iscc-r:nigrosinviolet.224','nigrosinviolet','nigrosin violet ',[86,60,92],'563c5c',5651548],
['nbs-iscc-r:nileblue.163','nileblue','nile blue',[102,173,164],'66ada4',6729124],
['nbs-iscc-r:nopalred.12','nopalred','nopal red',[188,63,74],'bc3f4a',12336970],
['nbs-iscc-r:ocherred.19','ocherred','ocher red',[144,93,93],'905d5d',9461085],
['nbs-iscc-r:ochraceousbuff.28','ochraceousbuff','ochraceous-buff',[244,194,194],'f4c2c2',16040642],
['nbs-iscc-r:ochraceousbuff.29','ochraceousbuff','ochraceous-buff',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-r:ochraceousorange.68','ochraceousorange','ochraceous-orange',[234,162,33],'eaa221',15376929],
['nbs-iscc-r:ochraceousorange.69','ochraceousorange','ochraceous-orange',[201,133,0],'c98500',13206784],
['nbs-iscc-r:ochraceoussalmon.53','ochraceoussalmon','ochraceous-salmon',[217,144,88],'d99058',14258264],
['nbs-iscc-r:ochraceoustawny.72','ochraceoustawny','ochraceous-tawny',[190,138,61],'be8a3d',12487229],
['nbs-iscc-r:ochraceoustawny.74','ochraceoustawny','ochraceous-tawny',[153,101,21],'996515',10052885],
['nbs-iscc-r:oilgreen.120','oilgreen','oil green',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-r:oldburgundy.21','oldburgundy','old burgundy',[46,29,33],'2e1d21',3022113],
['nbs-iscc-r:oldgold.88','oldgold','old gold',[171,145,68],'ab9144',11243844],
['nbs-iscc-r:oldgold.94','oldgold','old gold',[150,113,23],'967117',9859351],
['nbs-iscc-r:oldrose.15','oldrose','old rose',[171,78,82],'ab4e52',11226706],
['nbs-iscc-r:olive.107','olive','olive',[102,93,30],'665d1e',6708510],
['nbs-iscc-r:olive.110','olive','olive',[91,88,66],'5b5842',5986370],
['nbs-iscc-r:olivebrown.80','olivebrown','olive-brown ',[126,109,90],'7e6d5a',8285530],
['nbs-iscc-r:olivebuff.89','olivebuff','olive-buff',[243,229,171],'f3e5ab',15984043],
['nbs-iscc-r:olivebuff.90','olivebuff','olive-buff',[194,178,128],'c2b280',12759680],
['nbs-iscc-r:olivecitrine.107','olivecitrine','olive-citrine',[102,93,30],'665d1e',6708510],
['nbs-iscc-r:olivegray.109','olivegray','olive-gray ',[140,135,103],'8c8767',9209703],
['nbs-iscc-r:olivegray.112','olivegray','olive-gray ',[138,135,118],'8a8776',9078646],
['nbs-iscc-r:olivegreen.107','olivegreen','olive green',[102,93,30],'665d1e',6708510],
['nbs-iscc-r:olivelake.91','olivelake','olive lake',[161,143,96],'a18f60',10588000],
['nbs-iscc-r:olivelake.106','olivelake','olive lake',[134,126,54],'867e36',8814134],
['nbs-iscc-r:oliveocher.87','oliveocher','olive-ocher',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-r:oliveocher.88','oliveocher','olive-ocher',[171,145,68],'ab9144',11243844],
['nbs-iscc-r:oliveyellow.102','oliveyellow','olive-yellow ',[185,180,89],'b9b459',12170329],
['nbs-iscc-r:olivine.135','olivine','olivine',[147,197,146],'93c592',9684370],
['nbs-iscc-r:olympicblue.178','olympicblue','olympic blue',[0,103,165],'0067a5',26533],
['nbs-iscc-r:onionskinpink.29','onionskinpink','onion-skin pink',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-r:onionskinpink.53','onionskinpink','onion-skin pink',[217,144,88],'d99058',14258264],
['nbs-iscc-r:ontarioviolet.210','ontarioviolet','ontario violet',[140,130,182],'8c82b6',9208502],
['nbs-iscc-r:opalinegreen.119','opalinegreen','opaline green',[201,220,137],'c9dc89',13229193],
['nbs-iscc-r:opalinegreen.121','opalinegreen','opaline green',[218,223,183],'dadfb7',14344119],
['nbs-iscc-r:orange.50','orange','orange',[237,135,45],'ed872d',15566637],
['nbs-iscc-r:orangebuff.71','orangebuff','orange-buff ',[227,168,87],'e3a857',14919767],
['nbs-iscc-r:orangechrome.48','orangechrome','orange chrome',[243,132,0],'f38400',15959040],
['nbs-iscc-r:orangecinnamon.53','orangecinnamon','orange-cinnamon',[217,144,88],'d99058',14258264],
['nbs-iscc-r:orangecitrine.94','orangecitrine','orange-citrine',[150,113,23],'967117',9859351],
['nbs-iscc-r:orangepink.49','orangepink','orange-pink',[253,148,63],'fd943f',16618559],
['nbs-iscc-r:orangepink.52','orangepink','orange-pink',[250,181,127],'fab57f',16430463],
['nbs-iscc-r:orangerufous.51','orangerufous','orange-rufous ',[190,101,22],'be6516',12477718],
['nbs-iscc-r:orangevinaceous.3','orangevinaceous','orange-vinaceous ',[228,113,122],'e4717a',14971258],
['nbs-iscc-r:orangevinaceous.26','orangevinaceous','orange-vinaceous ',[248,131,121],'f88379',16286585],
['nbs-iscc-r:orientalgreen.136','orientalgreen','oriental green',[103,146,103],'679267',6787687],
['nbs-iscc-r:orientblue.182','orientblue','orient blue',[67,107,149],'436b95',4418453],
['nbs-iscc-r:orientblue.186','orientblue','orient blue',[83,104,120],'536878',5466232],
['nbs-iscc-r:orientpink.26','orientpink','orient pink',[248,131,121],'f88379',16286585],
['nbs-iscc-r:ouralgreen.134','ouralgreen','oural green',[182,229,175],'b6e5af',11986351],
['nbs-iscc-r:ouralgreen.135','ouralgreen','oural green',[147,197,146],'93c592',9684370],
['nbs-iscc-r:oxbloodred.13','oxbloodred','ox-blood red ',[132,27,45],'841b2d',8657709],
['nbs-iscc-r:oxideblue.178','oxideblue','oxide blue',[0,103,165],'0067a5',26533],
['nbs-iscc-r:oxideblue.182','oxideblue','oxide blue',[67,107,149],'436b95',4418453],
['nbs-iscc-r:paleamaranthpink.247','paleamaranthpink','pale amaranth pink',[230,143,172],'e68fac',15110060],
['nbs-iscc-r:paleamparoblue.184','paleamparoblue','pale amparo blue',[188,212,230],'bcd4e6',12375270],
['nbs-iscc-r:paleamparoblue.185','paleamparoblue','pale amparo blue',[145,163,176],'91a3b0',9544624],
['nbs-iscc-r:paleamparopurple.222','paleamparopurple','pale amparo purple',[182,149,192],'b695c0',11965888],
['nbs-iscc-r:paleanilinelilac.180','paleanilinelilac','pale aniline lilac',[161,202,241],'a1caf1',10603249],
['nbs-iscc-r:paleanilinelilac.181','paleanilinelilac','pale aniline lilac',[112,163,204],'70a3cc',7381964],
['nbs-iscc-r:paleblue.171','paleblue','pale blue',[156,209,220],'9cd1dc',10277340],
['nbs-iscc-r:paleblue.184','paleblue','pale blue',[188,212,230],'bcd4e6',12375270],
['nbs-iscc-r:palebluegreen.148','palebluegreen','pale blue-green',[199,230,215],'c7e6d7',13100759],
['nbs-iscc-r:paleblueviolet.195','paleblueviolet','pale blue-violet',[108,121,184],'6c79b8',7109048],
['nbs-iscc-r:palebluishlavender.213','palebluishlavender','pale bluish lavender ',[196,195,221],'c4c3dd',12895197],
['nbs-iscc-r:palebluishlavender.226','palebluishlavender','pale bluish lavender ',[214,202,221],'d6cadd',14076637],
['nbs-iscc-r:palebluishviolet.195','palebluishviolet','pale bluish violet',[108,121,184],'6c79b8',7109048],
['nbs-iscc-r:palebrownishdrab.10','palebrownishdrab','pale brownish drab',[193,182,179],'c1b6b3',12695219],
['nbs-iscc-r:palebrownishvinaceous.7','palebrownishvinaceous','pale brownish vinaceous',[234,216,215],'ead8d7',15390935],
['nbs-iscc-r:palebrownishvinaceous.8','palebrownishvinaceous','pale brownish vinaceous',[196,174,173],'c4aead',12889773],
['nbs-iscc-r:palecadetblue.181','palecadetblue','pale cadet blue',[112,163,204],'70a3cc',7381964],
['nbs-iscc-r:palecadetblue.185','palecadetblue','pale cadet blue',[145,163,176],'91a3b0',9544624],
['nbs-iscc-r:palecadetblue.199','palecadetblue','pale cadet blue',[135,145,191],'8791bf',8884671],
['nbs-iscc-r:palecadetblue.203','palecadetblue','pale cadet blue',[140,146,172],'8c92ac',9212588],
['nbs-iscc-r:palecampanulablue.198','palecampanulablue','pale campanula blue',[179,188,226],'b3bce2',11779298],
['nbs-iscc-r:palecampanulablue.199','palecampanulablue','pale campanula blue',[135,145,191],'8791bf',8884671],
['nbs-iscc-r:palecendregreen.119','palecendregreen','pale cendre green',[201,220,137],'c9dc89',13229193],
['nbs-iscc-r:palecendregreen.121','palecendregreen','pale cendre green',[218,223,183],'dadfb7',14344119],
['nbs-iscc-r:paleceruleanblue.180','paleceruleanblue','pale cerulean blue',[161,202,241],'a1caf1',10603249],
['nbs-iscc-r:paleceruleanblue.181','paleceruleanblue','pale cerulean blue',[112,163,204],'70a3cc',7381964],
['nbs-iscc-r:palechalcedonyyellow.104','palechalcedonyyellow','pale chalcedony yellow',[235,232,164],'ebe8a4',15460516],
['nbs-iscc-r:palecinnamonpink.28','palecinnamonpink','pale cinnamon-pink',[244,194,194],'f4c2c2',16040642],
['nbs-iscc-r:palecinnamonpink.29','palecinnamonpink','pale cinnamon-pink',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-r:palecinnamonpink.31','palecinnamonpink','pale cinnamon-pink',[236,213,197],'ecd5c5',15521221],
['nbs-iscc-r:palecinnamonpink.33','palecinnamonpink','pale cinnamon-pink',[194,172,153],'c2ac99',12758169],
['nbs-iscc-r:palecongopink.28','palecongopink','pale congo pink',[244,194,194],'f4c2c2',16040642],
['nbs-iscc-r:palecongopink.29','palecongopink','pale congo pink',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-r:palecongopink.31','palecongopink','pale congo pink',[236,213,197],'ecd5c5',15521221],
['nbs-iscc-r:palecongopink.32','palecongopink','pale congo pink',[199,173,163],'c7ada3',13086115],
['nbs-iscc-r:paledrabgray.263','paledrabgray','pale drab-gray',[242,243,244],'f2f3f4',15922164],
['nbs-iscc-r:paledrabgray.264','paledrabgray','pale drab-gray',[185,184,181],'b9b8b5',12171445],
['nbs-iscc-r:paledullglaucousblue.184','paledullglaucousblue','pale dull glaucous-blue',[188,212,230],'bcd4e6',12375270],
['nbs-iscc-r:paledullglaucousblue.185','paledullglaucousblue','pale dull glaucous-blue',[145,163,176],'91a3b0',9544624],
['nbs-iscc-r:paledullgreenyellow.119','paledullgreenyellow','pale dull green-yellow',[201,220,137],'c9dc89',13229193],
['nbs-iscc-r:paleecrudrab.10','paleecrudrab','pale ecru-drab',[193,182,179],'c1b6b3',12695219],
['nbs-iscc-r:palefleshcolor.28','palefleshcolor','pale flesh color',[244,194,194],'f4c2c2',16040642],
['nbs-iscc-r:palefleshcolor.29','palefleshcolor','pale flesh color',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-r:palefluoritegreen.148','palefluoritegreen','pale fluorite green',[199,230,215],'c7e6d7',13100759],
['nbs-iscc-r:paleglassgreen.119','paleglassgreen','pale glass green',[201,220,137],'c9dc89',13229193],
['nbs-iscc-r:paleglassgreen.121','paleglassgreen','pale glass green',[218,223,183],'dadfb7',14344119],
['nbs-iscc-r:paleglaucousblue.171','paleglaucousblue','pale glaucous-blue',[156,209,220],'9cd1dc',10277340],
['nbs-iscc-r:paleglaucousblue.184','paleglaucousblue','pale glaucous-blue',[188,212,230],'bcd4e6',12375270],
['nbs-iscc-r:paleglaucousgreen.148','paleglaucousgreen','pale glaucous-green ',[199,230,215],'c7e6d7',13100759],
['nbs-iscc-r:palegrayishblue.184','palegrayishblue','pale grayish blue',[188,212,230],'bcd4e6',12375270],
['nbs-iscc-r:palegrayishblue.202','palegrayishblue','pale grayish blue',[192,200,225],'c0c8e1',12634337],
['nbs-iscc-r:palegrayishblueviolet.202','palegrayishblueviolet','pale grayish blue-violet',[192,200,225],'c0c8e1',12634337],
['nbs-iscc-r:palegrayishvinaceous.10','palegrayishvinaceous','pale grayish vinaceous',[193,182,179],'c1b6b3',12695219],
['nbs-iscc-r:palegrayishvinaceous.31','palegrayishvinaceous','pale grayish vinaceous',[236,213,197],'ecd5c5',15521221],
['nbs-iscc-r:palegrayishvinaceous.32','palegrayishvinaceous','pale grayish vinaceous',[199,173,163],'c7ada3',13086115],
['nbs-iscc-r:palegrayishvioletblue.199','palegrayishvioletblue','pale grayish violet-blue',[135,145,191],'8791bf',8884671],
['nbs-iscc-r:palegreenbluegray.185','palegreenbluegray','pale green-blue gray',[145,163,176],'91a3b0',9544624],
['nbs-iscc-r:palegreenishyellow.101','palegreenishyellow','pale greenish yellow',[234,230,121],'eae679',15394425],
['nbs-iscc-r:palegreenishyellow.102','palegreenishyellow','pale greenish yellow',[185,180,89],'b9b459',12170329],
['nbs-iscc-r:palegreenyellow.101','palegreenyellow','pale green-yellow',[234,230,121],'eae679',15394425],
['nbs-iscc-r:palegreenyellow.104','palegreenyellow','pale green-yellow',[235,232,164],'ebe8a4',15460516],
['nbs-iscc-r:palegullgray.264','palegullgray','pale gull gray',[185,184,181],'b9b8b5',12171445],
['nbs-iscc-r:palehortenseviolet.221','palehortenseviolet','pale hortense violet',[213,186,219],'d5badb',14007003],
['nbs-iscc-r:palehortenseviolet.222','palehortenseviolet','pale hortense violet',[182,149,192],'b695c0',11965888],
['nbs-iscc-r:palekingsblue.213','palekingsblue','pale king\'s blue',[196,195,221],'c4c3dd',12895197],
['nbs-iscc-r:palelaeliapink.249','palelaeliapink','pale laelia pink ',[239,187,204],'efbbcc',15711180],
['nbs-iscc-r:palelaeliapink.250','palelaeliapink','pale laelia pink ',[213,151,174],'d597ae',13997998],
['nbs-iscc-r:palelavenderviolet.222','palelavenderviolet','pale lavender-violet',[182,149,192],'b695c0',11965888],
['nbs-iscc-r:palelavenderviolet.227','palelavenderviolet','pale lavender-violet',[170,152,169],'aa98a9',11180201],
['nbs-iscc-r:palelemonyellow.98','palelemonyellow','pale lemon yellow',[233,228,80],'e9e450',15328336],
['nbs-iscc-r:palelilac.252','palelilac','pale lilac',[232,204,215],'e8ccd7',15256791],
['nbs-iscc-r:palelobeliaviolet.226','palelobeliaviolet','pale lobelia violet',[214,202,221],'d6cadd',14076637],
['nbs-iscc-r:palelobeliaviolet.227','palelobeliaviolet','pale lobelia violet',[170,152,169],'aa98a9',11180201],
['nbs-iscc-r:palelumieregreen.119','palelumieregreen','pale lumiere green',[201,220,137],'c9dc89',13229193],
['nbs-iscc-r:palemauve.226','palemauve','pale mauve',[214,202,221],'d6cadd',14076637],
['nbs-iscc-r:palemazarineblue.184','palemazarineblue','pale mazarine blue',[188,212,230],'bcd4e6',12375270],
['nbs-iscc-r:palemazarineblue.185','palemazarineblue','pale mazarine blue',[145,163,176],'91a3b0',9544624],
['nbs-iscc-r:palemazarineblue.202','palemazarineblue','pale mazarine blue',[192,200,225],'c0c8e1',12634337],
['nbs-iscc-r:palemazarineblue.203','palemazarineblue','pale mazarine blue',[140,146,172],'8c92ac',9212588],
['nbs-iscc-r:palemediciblue.185','palemediciblue','pale medici blue',[145,163,176],'91a3b0',9544624],
['nbs-iscc-r:palemediciblue.190','palemediciblue','pale medici blue',[180,188,192],'b4bcc0',11844800],
['nbs-iscc-r:palemediciblue.191','palemediciblue','pale medici blue',[129,135,139],'81878b',8488843],
['nbs-iscc-r:palemethylblue.181','palemethylblue','pale methyl blue ',[112,163,204],'70a3cc',7381964],
['nbs-iscc-r:palemousegray.63','palemousegray','pale mouse gray ',[142,130,121],'8e8279',9339513],
['nbs-iscc-r:palemousegray.93','palemousegray','pale mouse gray ',[191,184,165],'bfb8a5',12564645],
['nbs-iscc-r:paleneropalinblue.184','paleneropalinblue','pale neropalin blue',[188,212,230],'bcd4e6',12375270],
['nbs-iscc-r:paleneutralgray.265','paleneutralgray','pale neutral gray',[132,132,130],'848482',8684674],
['nbs-iscc-r:paleniagaragreen.148','paleniagaragreen','pale niagara green',[199,230,215],'c7e6d7',13100759],
['nbs-iscc-r:palenileblue.162','palenileblue','pale nile blue',[150,222,209],'96ded1',9887441],
['nbs-iscc-r:paleochraceousbuff.28','paleochraceousbuff','pale ochraceous-buff',[244,194,194],'f4c2c2',16040642],
['nbs-iscc-r:paleochraceoussalmon.73','paleochraceoussalmon','pale ochraceous-salmon',[250,214,165],'fad6a5',16438949],
['nbs-iscc-r:paleochraceoussalmon.89','paleochraceoussalmon','pale ochraceous-salmon',[243,229,171],'f3e5ab',15984043],
['nbs-iscc-r:paleolivebuff.89','paleolivebuff','pale olive-buff',[243,229,171],'f3e5ab',15984043],
['nbs-iscc-r:paleolivegray.92','paleolivegray','pale olive-gray',[240,234,214],'f0ead6',15788758],
['nbs-iscc-r:paleolivegray.93','paleolivegray','pale olive-gray',[191,184,165],'bfb8a5',12564645],
['nbs-iscc-r:paleolivine.148','paleolivine','pale olivine',[199,230,215],'c7e6d7',13100759],
['nbs-iscc-r:paleorangeyellow.70','paleorangeyellow','pale orange-yellow',[251,201,127],'fbc97f',16501119],
['nbs-iscc-r:paleorangeyellow.71','paleorangeyellow','pale orange-yellow',[227,168,87],'e3a857',14919767],
['nbs-iscc-r:paleorangeyellow.83','paleorangeyellow','pale orange-yellow',[250,218,94],'fada5e',16439902],
['nbs-iscc-r:paleorangeyellow.84','paleorangeyellow','pale orange-yellow',[212,175,55],'d4af37',13938487],
['nbs-iscc-r:paleorangeyellow.86','paleorangeyellow','pale orange-yellow',[248,222,126],'f8de7e',16309886],
['nbs-iscc-r:paleorangeyellow.87','paleorangeyellow','pale orange-yellow',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-r:palepaynesgray.185','palepaynesgray','pale payne\'s gray',[145,163,176],'91a3b0',9544624],
['nbs-iscc-r:palepersianlilac.250','palepersianlilac','pale persian lilac',[213,151,174],'d597ae',13997998],
['nbs-iscc-r:palepinkishbuff.73','palepinkishbuff','pale pinkish buff',[250,214,165],'fad6a5',16438949],
['nbs-iscc-r:palepinkishcinnamon.73','palepinkishcinnamon','pale pinkish cinnamon',[250,214,165],'fad6a5',16438949],
['nbs-iscc-r:palepurpledrab.253','palepurpledrab','pale purple-drab',[195,166,177],'c3a6b1',12822193],
['nbs-iscc-r:palepurplishgray.233','palepurplishgray','pale purplish gray',[139,133,137],'8b8589',9143689],
['nbs-iscc-r:palepurplishvinaceous.4','palepurplishvinaceous','pale purplish vinaceous',[249,204,202],'f9ccca',16370890],
['nbs-iscc-r:palepurplishvinaceous.5','palepurplishvinaceous','pale purplish vinaceous',[222,165,164],'dea5a4',14591396],
['nbs-iscc-r:palequakerdrab.232','palequakerdrab','pale quaker drab',[191,185,189],'bfb9bd',12564925],
['nbs-iscc-r:palequakerdrab.233','palequakerdrab','pale quaker drab',[139,133,137],'8b8589',9143689],
['nbs-iscc-r:palerhodonitepink.249','palerhodonitepink','pale rhodonite pink',[239,187,204],'efbbcc',15711180],
['nbs-iscc-r:palerosepurple.222','palerosepurple','pale rose-purple ',[182,149,192],'b695c0',11965888],
['nbs-iscc-r:palerosolanepurple.248','palerosolanepurple','pale rosolane purple',[222,111,161],'de6fa1',14577569],
['nbs-iscc-r:palerussianblue.184','palerussianblue','pale russian blue',[188,212,230],'bcd4e6',12375270],
['nbs-iscc-r:palesalmoncolor.73','palesalmoncolor','pale salmon color',[250,214,165],'fad6a5',16438949],
['nbs-iscc-r:palesmokegray.93','palesmokegray','pale smoke gray',[191,184,165],'bfb8a5',12564645],
['nbs-iscc-r:palesoftblueviolet.199','palesoftblueviolet','pale soft blue-violet',[135,145,191],'8791bf',8884671],
['nbs-iscc-r:palesoftblueviolet.210','palesoftblueviolet','pale soft blue-violet',[140,130,182],'8c82b6',9208502],
['nbs-iscc-r:palesulphategreen.163','palesulphategreen','pale sulphate green',[102,173,164],'66ada4',6729124],
['nbs-iscc-r:paletibergreen.119','paletibergreen','pale tiber green',[201,220,137],'c9dc89',13229193],
['nbs-iscc-r:paleturquoisegreen.162','paleturquoisegreen','pale turquoise green',[150,222,209],'96ded1',9887441],
['nbs-iscc-r:paleturtlegreen.119','paleturtlegreen','pale turtle green',[201,220,137],'c9dc89',13229193],
['nbs-iscc-r:paleturtlegreen.121','paleturtlegreen','pale turtle green',[218,223,183],'dadfb7',14344119],
['nbs-iscc-r:palevarleysgray.214','palevarleysgray','pale varley\'s gray ',[150,144,171],'9690ab',9867435],
['nbs-iscc-r:palevarleysgray.227','palevarleysgray','pale varley\'s gray ',[170,152,169],'aa98a9',11180201],
['nbs-iscc-r:paleverbenaviolet.209','paleverbenaviolet','pale verbena violet',[220,208,255],'dcd0ff',14471423],
['nbs-iscc-r:paleveronesegreen.119','paleveronesegreen','pale veronese green',[201,220,137],'c9dc89',13229193],
['nbs-iscc-r:paleveronesegreen.121','paleveronesegreen','pale veronese green',[218,223,183],'dadfb7',14344119],
['nbs-iscc-r:palevinaceous.5','palevinaceous','pale vinaceous',[222,165,164],'dea5a4',14591396],
['nbs-iscc-r:palevinaceousdrab.8','palevinaceousdrab','pale vinaceous-drab',[196,174,173],'c4aead',12889773],
['nbs-iscc-r:palevinaceousdrab.10','palevinaceousdrab','pale vinaceous-drab',[193,182,179],'c1b6b3',12695219],
['nbs-iscc-r:palevinaceousdrab.18','palevinaceousdrab','pale vinaceous-drab',[173,136,132],'ad8884',11372676],
['nbs-iscc-r:palevinaceousdrab.22','palevinaceousdrab','pale vinaceous-drab',[143,129,127],'8f817f',9404799],
['nbs-iscc-r:palevinaceousfawn.10','palevinaceousfawn','pale vinaceous-fawn',[193,182,179],'c1b6b3',12695219],
['nbs-iscc-r:palevinaceousfawn.31','palevinaceousfawn','pale vinaceous-fawn',[236,213,197],'ecd5c5',15521221],
['nbs-iscc-r:palevinaceousfawn.32','palevinaceousfawn','pale vinaceous-fawn',[199,173,163],'c7ada3',13086115],
['nbs-iscc-r:palevinaceousfawn.33','palevinaceousfawn','pale vinaceous-fawn',[194,172,153],'c2ac99',12758169],
['nbs-iscc-r:palevinaceouslilac.249','palevinaceouslilac','pale vinaceous-lilac',[239,187,204],'efbbcc',15711180],
['nbs-iscc-r:palevinaceouslilac.252','palevinaceouslilac','pale vinaceous-lilac',[232,204,215],'e8ccd7',15256791],
['nbs-iscc-r:palevinaceouspink.31','palevinaceouspink','pale vinaceous-pink',[236,213,197],'ecd5c5',15521221],
['nbs-iscc-r:palevinaceouspink.32','palevinaceouspink','pale vinaceous-pink',[199,173,163],'c7ada3',13086115],
['nbs-iscc-r:paleviolet.199','paleviolet','pale violet',[135,145,191],'8791bf',8884671],
['nbs-iscc-r:paleviolet.210','paleviolet','pale violet',[140,130,182],'8c82b6',9208502],
['nbs-iscc-r:palevioletblue.177','palevioletblue','pale violet-blue',[73,151,208],'4997d0',4822992],
['nbs-iscc-r:palevioletgray.233','palevioletgray','pale violet-gray',[139,133,137],'8b8589',9143689],
['nbs-iscc-r:palevioletgray.265','palevioletgray','pale violet-gray',[132,132,130],'848482',8684674],
['nbs-iscc-r:palevioletplumbeous.185','palevioletplumbeous','pale violet-plumbeous ',[145,163,176],'91a3b0',9544624],
['nbs-iscc-r:palevioletplumbeous.203','palevioletplumbeous','pale violet-plumbeous ',[140,146,172],'8c92ac',9212588],
['nbs-iscc-r:paleviridineyellow.101','paleviridineyellow','pale viridine yellow',[234,230,121],'eae679',15394425],
['nbs-iscc-r:paleviridineyellow.104','paleviridineyellow','pale viridine yellow',[235,232,164],'ebe8a4',15460516],
['nbs-iscc-r:palewindsorblue.181','palewindsorblue','pale windsor blue',[112,163,204],'70a3cc',7381964],
['nbs-iscc-r:palewindsorblue.199','palewindsorblue','pale windsor blue',[135,145,191],'8791bf',8884671],
['nbs-iscc-r:palewistariablue.209','palewistariablue','pale wistaria blue',[220,208,255],'dcd0ff',14471423],
['nbs-iscc-r:palewistariablue.210','palewistariablue','pale wistaria blue',[140,130,182],'8c82b6',9208502],
['nbs-iscc-r:palewistariaviolet.210','palewistariaviolet','pale wistaria violet',[140,130,182],'8c82b6',9208502],
['nbs-iscc-r:palewistariaviolet.214','palewistariaviolet','pale wistaria violet',[150,144,171],'9690ab',9867435],
['nbs-iscc-r:paleyellowgreen.119','paleyellowgreen','pale yellow-green',[201,220,137],'c9dc89',13229193],
['nbs-iscc-r:paleyelloworange.28','paleyelloworange','pale yellow-orange',[244,194,194],'f4c2c2',16040642],
['nbs-iscc-r:pallidblueviolet.198','pallidblueviolet','pallid blue-violet',[179,188,226],'b3bce2',11779298],
['nbs-iscc-r:pallidblueviolet.199','pallidblueviolet','pallid blue-violet',[135,145,191],'8791bf',8884671],
['nbs-iscc-r:pallidbluishviolet.198','pallidbluishviolet','pallid bluish violet',[179,188,226],'b3bce2',11779298],
['nbs-iscc-r:pallidbluishviolet.199','pallidbluishviolet','pallid bluish violet',[135,145,191],'8791bf',8884671],
['nbs-iscc-r:pallidbluishviolet.209','pallidbluishviolet','pallid bluish violet',[220,208,255],'dcd0ff',14471423],
['nbs-iscc-r:pallidbluishviolet.210','pallidbluishviolet','pallid bluish violet',[140,130,182],'8c82b6',9208502],
['nbs-iscc-r:pallidbrownishdrab.10','pallidbrownishdrab','pallid brownish drab',[193,182,179],'c1b6b3',12695219],
['nbs-iscc-r:pallidgrayishvioletblue.199','pallidgrayishvioletblue','pallid grayish violet-blue',[135,145,191],'8791bf',8884671],
['nbs-iscc-r:pallidgrayishvioletblue.203','pallidgrayishvioletblue','pallid grayish violet-blue',[140,146,172],'8c92ac',9212588],
['nbs-iscc-r:pallidmethylblue.180','pallidmethylblue','pallid methyl blue ',[161,202,241],'a1caf1',10603249],
['nbs-iscc-r:pallidmousegray.10','pallidmousegray','pallid mouse gray ',[193,182,179],'c1b6b3',12695219],
['nbs-iscc-r:pallidmousegray.264','pallidmousegray','pallid mouse gray ',[185,184,181],'b9b8b5',12171445],
['nbs-iscc-r:pallidneutralgray.264','pallidneutralgray','pallid neutral gray',[185,184,181],'b9b8b5',12171445],
['nbs-iscc-r:pallidpurpledrab.232','pallidpurpledrab','pallid purple-drab',[191,185,189],'bfb9bd',12564925],
['nbs-iscc-r:pallidpurpledrab.253','pallidpurpledrab','pallid purple-drab',[195,166,177],'c3a6b1',12822193],
['nbs-iscc-r:pallidpurplishgray.232','pallidpurplishgray','pallid purplish gray ',[191,185,189],'bfb9bd',12564925],
['nbs-iscc-r:pallidquakerdrab.232','pallidquakerdrab','pallid quaker drab',[191,185,189],'bfb9bd',12564925],
['nbs-iscc-r:pallidsoftblueviolet.198','pallidsoftblueviolet','pallid soft blue-violet ',[179,188,226],'b3bce2',11779298],
['nbs-iscc-r:pallidsoftblueviolet.199','pallidsoftblueviolet','pallid soft blue-violet ',[135,145,191],'8791bf',8884671],
['nbs-iscc-r:pallidsoftblueviolet.209','pallidsoftblueviolet','pallid soft blue-violet ',[220,208,255],'dcd0ff',14471423],
['nbs-iscc-r:pallidsoftblueviolet.210','pallidsoftblueviolet','pallid soft blue-violet ',[140,130,182],'8c82b6',9208502],
['nbs-iscc-r:pallidvinaceousdrab.232','pallidvinaceousdrab','pallid vinaceous-drab',[191,185,189],'bfb9bd',12564925],
['nbs-iscc-r:pallidvinaceousdrab.252','pallidvinaceousdrab','pallid vinaceous-drab',[232,204,215],'e8ccd7',15256791],
['nbs-iscc-r:pallidviolet.198','pallidviolet','pallid violet',[179,188,226],'b3bce2',11779298],
['nbs-iscc-r:pallidviolet.199','pallidviolet','pallid violet',[135,145,191],'8791bf',8884671],
['nbs-iscc-r:pallidviolet.202','pallidviolet','pallid violet',[192,200,225],'c0c8e1',12634337],
['nbs-iscc-r:pallidviolet.203','pallidviolet','pallid violet',[140,146,172],'8c92ac',9212588],
['nbs-iscc-r:pallidviolet.209','pallidviolet','pallid violet',[220,208,255],'dcd0ff',14471423],
['nbs-iscc-r:pallidviolet.210','pallidviolet','pallid violet',[140,130,182],'8c82b6',9208502],
['nbs-iscc-r:pallidviolet.213','pallidviolet','pallid violet',[196,195,221],'c4c3dd',12895197],
['nbs-iscc-r:pallidviolet.214','pallidviolet','pallid violet',[150,144,171],'9690ab',9867435],
['nbs-iscc-r:pallidvioletblue.198','pallidvioletblue','pallid violet-blue',[179,188,226],'b3bce2',11779298],
['nbs-iscc-r:pallidvioletblue.199','pallidvioletblue','pallid violet-blue',[135,145,191],'8791bf',8884671],
['nbs-iscc-r:pansypurple.259','pansypurple','pansy purple',[103,49,71],'673147',6762823],
['nbs-iscc-r:pansyviolet.218','pansyviolet','pansy violet',[135,86,146],'875692',8869522],
['nbs-iscc-r:parisblue.178','parisblue','paris blue',[0,103,165],'0067a5',26533],
['nbs-iscc-r:parisblue.179','parisblue','paris blue',[0,65,106],'00416a',16746],
['nbs-iscc-r:parisgreen.135','parisgreen','paris green',[147,197,146],'93c592',9684370],
['nbs-iscc-r:parrotgreen.120','parrotgreen','parrot green',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-r:parulablue.185','parulablue','parula blue',[145,163,176],'91a3b0',9544624],
['nbs-iscc-r:parulablue.186','parulablue','parula blue',[83,104,120],'536878',5466232],
['nbs-iscc-r:patentblue.174','patentblue','patent blue',[0,73,88],'004958',18776],
['nbs-iscc-r:paynesgray.186','paynesgray','payne\'s gray',[83,104,120],'536878',5466232],
['nbs-iscc-r:peachred.37','peachred','peach red',[203,109,81],'cb6d51',13331793],
['nbs-iscc-r:peacockblue.169','peacockblue','peacock blue',[0,119,145],'007791',30609],
['nbs-iscc-r:peacockgreen.120','peacockgreen','peacock green',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-r:peagreen.155','peagreen','pea green',[125,137,132],'7d8984',8227204],
['nbs-iscc-r:pearlblue.202','pearlblue','pearl blue',[192,200,225],'c0c8e1',12634337],
['nbs-iscc-r:pearlblue.203','pearlblue','pearl blue',[140,146,172],'8c92ac',9212588],
['nbs-iscc-r:pearlgray.264','pearlgray','pearl gray',[185,184,181],'b9b8b5',12171445],
['nbs-iscc-r:pecanbrown.57','pecanbrown','pecan brown',[166,123,91],'a67b5b',10910555],
['nbs-iscc-r:perillapurple.259','perillapurple','perilla purple',[103,49,71],'673147',6762823],
['nbs-iscc-r:persianblue.184','persianblue','persian blue',[188,212,230],'bcd4e6',12375270],
['nbs-iscc-r:persianblue.185','persianblue','persian blue',[145,163,176],'91a3b0',9544624],
['nbs-iscc-r:persianlilac.251','persianlilac','persian lilac',[193,126,145],'c17e91',12680849],
['nbs-iscc-r:petuniaviolet.219','petuniaviolet','petunia violet',[96,47,107],'602f6b',6303595],
['nbs-iscc-r:phenylblue.194','phenylblue','phenyl blue',[48,38,122],'30267a',3155578],
['nbs-iscc-r:phloxpink.222','phloxpink','phlox pink',[182,149,192],'b695c0',11965888],
['nbs-iscc-r:phloxpurple.237','phloxpurple','phlox purple',[158,79,136],'9e4f88',10375048],
['nbs-iscc-r:picricyellow.98','picricyellow','picric yellow',[233,228,80],'e9e450',15328336],
['nbs-iscc-r:pinardyellow.83','pinardyellow','pinard yellow',[250,218,94],'fada5e',16439902],
['nbs-iscc-r:pinardyellow.86','pinardyellow','pinard yellow',[248,222,126],'f8de7e',16309886],
['nbs-iscc-r:pinkishbuff.28','pinkishbuff','pinkish buff',[244,194,194],'f4c2c2',16040642],
['nbs-iscc-r:pinkishbuff.29','pinkishbuff','pinkish buff',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-r:pinkishcinnamon.71','pinkishcinnamon','pinkish cinnamon',[227,168,87],'e3a857',14919767],
['nbs-iscc-r:pinkishcinnamon.76','pinkishcinnamon','pinkish cinnamon',[193,154,107],'c19a6b',12687979],
['nbs-iscc-r:pinkishvinaceous.29','pinkishvinaceous','pinkish vinaceous',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-r:pistachiogreen.136','pistachiogreen','pistachio green',[103,146,103],'679267',6787687],
['nbs-iscc-r:pleromaviolet.206','pleromaviolet','pleroma violet',[126,115,184],'7e73b8',8287160],
['nbs-iscc-r:pleromaviolet.207','pleromaviolet','pleroma violet',[96,78,151],'604e97',6311575],
['nbs-iscc-r:plumbagoblue.214','plumbagoblue','plumbago blue',[150,144,171],'9690ab',9867435],
['nbs-iscc-r:plumbagoblue.227','plumbagoblue','plumbago blue',[170,152,169],'aa98a9',11180201],
['nbs-iscc-r:plumbagogray.233','plumbagogray','plumbago gray',[139,133,137],'8b8589',9143689],
['nbs-iscc-r:plumbeous.185','plumbeous','plumbeous',[145,163,176],'91a3b0',9544624],
['nbs-iscc-r:plumbeous.191','plumbeous','plumbeous',[129,135,139],'81878b',8488843],
['nbs-iscc-r:plumbeousblack.187','plumbeousblack','plumbeous black ',[54,69,79],'36454f',3556687],
['nbs-iscc-r:plumbeousblack.267','plumbeousblack','plumbeous black ',[34,34,34],'222222',2236962],
['nbs-iscc-r:plumpurple.212','plumpurple','plum purple',[47,33,64],'2f2140',3088704],
['nbs-iscc-r:poisgreen.122','poisgreen','pois green',[143,151,121],'8f9779',9410425],
['nbs-iscc-r:pompeianred.15','pompeianred','pompeian red',[171,78,82],'ab4e52',11226706],
['nbs-iscc-r:porcelainblue.172','porcelainblue','porcelain blue',[102,170,188],'66aabc',6728380],
['nbs-iscc-r:porcelaingreen.164','porcelaingreen','porcelain green',[49,120,115],'317873',3242099],
['nbs-iscc-r:primroseyellow.101','primroseyellow','primrose yellow',[234,230,121],'eae679',15394425],
['nbs-iscc-r:primulineyellow.84','primulineyellow','primuline yellow',[212,175,55],'d4af37',13938487],
['nbs-iscc-r:proutsbrown.58','proutsbrown','prout\'s brown',[111,78,55],'6f4e37',7294519],
['nbs-iscc-r:prunepurple.224','prunepurple','prune purple',[86,60,92],'563c5c',5651548],
['nbs-iscc-r:prussianblue.178','prussianblue','prussian blue',[0,103,165],'0067a5',26533],
['nbs-iscc-r:prussianblue.182','prussianblue','prussian blue',[67,107,149],'436b95',4418453],
['nbs-iscc-r:prussiangreen.165','prussiangreen','prussian green',[0,75,73],'004b49',19273],
['nbs-iscc-r:prussianred.19','prussianred','prussian red',[144,93,93],'905d5d',9461085],
['nbs-iscc-r:puritangray.191','puritangray','puritan gray',[129,135,139],'81878b',8488843],
['nbs-iscc-r:purple.236','purple','purple',[135,0,116],'870074',8847476],
['nbs-iscc-r:purpledrab.228','purpledrab','purple-drab',[121,104,120],'796878',7956600],
['nbs-iscc-r:purplishgray.228','purplishgray','purplish gray',[121,104,120],'796878',7956600],
['nbs-iscc-r:purplishgray.234','purplishgray','purplish gray',[93,85,91],'5d555b',6116699],
['nbs-iscc-r:purplishlilac.240','purplishlilac','purplish lilac',[183,132,167],'b784a7',12027047],
['nbs-iscc-r:purplishlilac.250','purplishlilac','purplish lilac',[213,151,174],'d597ae',13997998],
['nbs-iscc-r:purplishvinaceous.18','purplishvinaceous','purplish vinaceous',[173,136,132],'ad8884',11372676],
['nbs-iscc-r:pyriteyellow.103','pyriteyellow','pyrite yellow',[152,148,62],'98943e',9999422],
['nbs-iscc-r:quakerdrab.233','quakerdrab','quaker drab',[139,133,137],'8b8589',9143689],
['nbs-iscc-r:quakerdrab.265','quakerdrab','quaker drab',[132,132,130],'848482',8684674],
['nbs-iscc-r:rainettegreen.120','rainettegreen','rainette green',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-r:raisinblack.229','raisinblack','raisin black',[80,64,77],'50404d',5259341],
['nbs-iscc-r:raisinpurple.242','raisinpurple','raisin purple',[93,57,84],'5d3954',6109524],
['nbs-iscc-r:ramierblue.214','ramierblue','ramier blue',[150,144,171],'9690ab',9867435],
['nbs-iscc-r:ramierblue.215','ramierblue','ramier blue',[85,76,105],'554c69',5590121],
['nbs-iscc-r:ramierblue.228','ramierblue','ramier blue',[121,104,120],'796878',7956600],
['nbs-iscc-r:rawsienna.72','rawsienna','raw sienna',[190,138,61],'be8a3d',12487229],
['nbs-iscc-r:rawsienna.74','rawsienna','raw sienna',[153,101,21],'996515',10052885],
['nbs-iscc-r:reedyellow.102','reedyellow','reed yellow',[185,180,89],'b9b459',12170329],
['nbs-iscc-r:rejanegreen.135','rejanegreen','rejane green',[147,197,146],'93c592',9684370],
['nbs-iscc-r:rejanegreen.136','rejanegreen','rejane green',[103,146,103],'679267',6787687],
['nbs-iscc-r:rhodaminepurple.254','rhodaminepurple','rhodamine purple',[206,70,118],'ce4676',13518454],
['nbs-iscc-r:rhodonitepink.251','rhodonitepink','rhodonite pink',[193,126,145],'c17e91',12680849],
['nbs-iscc-r:rinnemannsgreen.120','rinnemannsgreen','rinnemann\'s green',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-r:rivagegreen.119','rivagegreen','rivage green',[201,220,137],'c9dc89',13229193],
['nbs-iscc-r:rivagegreen.120','rivagegreen','rivage green',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-r:rocellinpurple.248','rocellinpurple','rocellin purple',[222,111,161],'de6fa1',14577569],
['nbs-iscc-r:rocellinpurple.251','rocellinpurple','rocellin purple',[193,126,145],'c17e91',12680849],
['nbs-iscc-r:romangreen.107','romangreen','roman green',[102,93,30],'665d1e',6708510],
['nbs-iscc-r:roodsblue.176','roodsblue','rood\'s blue',[0,161,194],'00a1c2',41410],
['nbs-iscc-r:roodsbrown.45','roodsbrown','rood\'s brown',[151,127,115],'977f73',9928563],
['nbs-iscc-r:roodsbrown.57','roodsbrown','rood\'s brown',[166,123,91],'a67b5b',10910555],
['nbs-iscc-r:roodsbrown.58','roodsbrown','rood\'s brown',[111,78,55],'6f4e37',7294519],
['nbs-iscc-r:roodslavender.227','roodslavender','rood\'s lavender',[170,152,169],'aa98a9',11180201],
['nbs-iscc-r:roodsviolet.238','roodsviolet','rood\'s violet',[112,41,99],'702963',7350627],
['nbs-iscc-r:rosecolor.248','rosecolor','rose color',[222,111,161],'de6fa1',14577569],
['nbs-iscc-r:rosecolor.254','rosecolor','rose color',[206,70,118],'ce4676',13518454],
['nbs-iscc-r:rosedoree.27','rosedoree','rose doree ',[230,103,97],'e66761',15099745],
['nbs-iscc-r:rosepink.249','rosepink','rose pink',[239,187,204],'efbbcc',15711180],
['nbs-iscc-r:rosepurple.217','rosepurple','rose-purple ',[211,153,230],'d399e6',13867494],
['nbs-iscc-r:rosered.11','rosered','rose red',[190,0,50],'be0032',12451890],
['nbs-iscc-r:roslynblue.197','roslynblue','roslyn blue',[39,36,88],'272458',2565208],
['nbs-iscc-r:roslynblue.200','roslynblue','roslyn blue',[78,81,128],'4e5180',5132672],
['nbs-iscc-r:roslynblue.211','roslynblue','roslyn blue',[96,78,129],'604e81',6311553],
['nbs-iscc-r:rosolanepink.246','rosolanepink','rosolane pink',[255,200,214],'ffc8d6',16763094],
['nbs-iscc-r:rosolanepink.249','rosolanepink','rosolane pink',[239,187,204],'efbbcc',15711180],
['nbs-iscc-r:rosolanepurple.255','rosolanepurple','rosolane purple',[179,68,108],'b3446c',11748460],
['nbs-iscc-r:royalpurple.211','royalpurple','royal purple',[96,78,129],'604e81',6311553],
['nbs-iscc-r:rufous.37','rufous','rufous',[203,109,81],'cb6d51',13331793],
['nbs-iscc-r:russet.58','russet','russet',[111,78,55],'6f4e37',7294519],
['nbs-iscc-r:russetvinaceous.19','russetvinaceous','russet-vinaceous',[144,93,93],'905d5d',9461085],
['nbs-iscc-r:russetvinaceous.42','russetvinaceous','russet-vinaceous',[168,124,109],'a87c6d',11041901],
['nbs-iscc-r:russianblue.184','russianblue','russian blue',[188,212,230],'bcd4e6',12375270],
['nbs-iscc-r:russianblue.185','russianblue','russian blue',[145,163,176],'91a3b0',9544624],
['nbs-iscc-r:russiangreen.150','russiangreen','russian green',[94,113,106],'5e716a',6189418],
['nbs-iscc-r:saccardosolive.94','saccardosolive','saccardo\'s olive',[150,113,23],'967117',9859351],
['nbs-iscc-r:saccardosolive.95','saccardosolive','saccardo\'s olive',[108,84,30],'6c541e',7099422],
['nbs-iscc-r:saccardosslate.192','saccardosslate','saccardo\'s slate',[81,88,94],'51585e',5331038],
['nbs-iscc-r:saccardosumber.80','saccardosumber','saccardo\'s umber',[126,109,90],'7e6d5a',8285530],
['nbs-iscc-r:saccardosumber.95','saccardosumber','saccardo\'s umber',[108,84,30],'6c541e',7099422],
['nbs-iscc-r:saccardosviolet.218','saccardosviolet','saccardo\'s violet',[135,86,146],'875692',8869522],
['nbs-iscc-r:saccardosviolet.223','saccardosviolet','saccardo\'s violet',[134,96,142],'86608e',8806542],
['nbs-iscc-r:safranopink.26','safranopink','safrano pink',[248,131,121],'f88379',16286585],
['nbs-iscc-r:sagegreen.149','sagegreen','sage green',[141,163,153],'8da399',9282457],
['nbs-iscc-r:sailorblue.196','sailorblue','sailor blue',[84,90,167],'545aa7',5528231],
['nbs-iscc-r:salmonbuff.71','salmonbuff','salmon-buff ',[227,168,87],'e3a857',14919767],
['nbs-iscc-r:salmonbuff.73','salmonbuff','salmon-buff ',[250,214,165],'fad6a5',16438949],
['nbs-iscc-r:salmoncolor.29','salmoncolor','salmon color',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-r:salmoncolor.52','salmoncolor','salmon color',[250,181,127],'fab57f',16430463],
['nbs-iscc-r:salmoncolor.53','salmoncolor','salmon color',[217,144,88],'d99058',14258264],
['nbs-iscc-r:salmonorange.50','salmonorange','salmon-orange',[237,135,45],'ed872d',15566637],
['nbs-iscc-r:salviablue.194','salviablue','salvia blue',[48,38,122],'30267a',3155578],
['nbs-iscc-r:sanfordsbrown.54','sanfordsbrown','sanford\'s brown',[174,105,56],'ae6938',11430200],
['nbs-iscc-r:sanfordsbrown.55','sanfordsbrown','sanford\'s brown',[128,70,27],'80461b',8406555],
['nbs-iscc-r:sayalbrown.74','sayalbrown','sayal brown',[153,101,21],'996515',10052885],
['nbs-iscc-r:sayalbrown.77','sayalbrown','sayal brown',[130,102,68],'826644',8545860],
['nbs-iscc-r:scarlet.35','scarlet','scarlet',[217,96,59],'d9603b',14245947],
['nbs-iscc-r:scarletred.11','scarletred','scarlet-red ',[190,0,50],'be0032',12451890],
['nbs-iscc-r:scheelesgreen.117','scheelesgreen','scheele\'s green',[126,159,46],'7e9f2e',8298286],
['nbs-iscc-r:scheelesgreen.120','scheelesgreen','scheele\'s green',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-r:schoenfeldspurple.256','schoenfeldspurple','schoenfeld\'s purple',[120,24,74],'78184a',7870538],
['nbs-iscc-r:seafoamgreen.104','seafoamgreen','sea-foam green ',[235,232,164],'ebe8a4',15460516],
['nbs-iscc-r:seafoamyellow.101','seafoamyellow','sea-foam yellow ',[234,230,121],'eae679',15394425],
['nbs-iscc-r:seafoamyellow.104','seafoamyellow','sea-foam yellow ',[235,232,164],'ebe8a4',15460516],
['nbs-iscc-r:seagreen.164','seagreen','sea green',[49,120,115],'317873',3242099],
['nbs-iscc-r:sealbrown.23','sealbrown','seal brown',[92,80,79],'5c504f',6049871],
['nbs-iscc-r:sealbrown.46','sealbrown','seal brown',[103,76,71],'674c47',6769735],
['nbs-iscc-r:seashellpink.31','seashellpink','seashell pink',[236,213,197],'ecd5c5',15521221],
['nbs-iscc-r:seashellpink.33','seashellpink','seashell pink',[194,172,153],'c2ac99',12758169],
['nbs-iscc-r:seashellpink.73','seashellpink','seashell pink',[250,214,165],'fad6a5',16438949],
['nbs-iscc-r:sepia.81','sepia','sepia',[72,60,50],'483c32',4734002],
['nbs-iscc-r:sepia.95','sepia','sepia',[108,84,30],'6c541e',7099422],
['nbs-iscc-r:serpentinegreen.106','serpentinegreen','serpentine green',[134,126,54],'867e36',8814134],
['nbs-iscc-r:shamrockgreen.136','shamrockgreen','shamrock green',[103,146,103],'679267',6787687],
['nbs-iscc-r:shellpink.31','shellpink','shell pink',[236,213,197],'ecd5c5',15521221],
['nbs-iscc-r:shellpink.32','shellpink','shell pink',[199,173,163],'c7ada3',13086115],
['nbs-iscc-r:shrimppink.28','shrimppink','shrimp pink',[244,194,194],'f4c2c2',16040642],
['nbs-iscc-r:shrimppink.29','shrimppink','shrimp pink',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-r:skobeloffgreen.160','skobeloffgreen','skobeloff green',[0,122,116],'007a74',31348],
['nbs-iscc-r:skyblue.181','skyblue','sky blue',[112,163,204],'70a3cc',7381964],
['nbs-iscc-r:skygray.184','skygray','sky gray',[188,212,230],'bcd4e6',12375270],
['nbs-iscc-r:slateblack.267','slateblack','slate-black',[34,34,34],'222222',2236962],
['nbs-iscc-r:slateblue.204','slateblue','slate-blue ',[76,81,109],'4c516d',5001581],
['nbs-iscc-r:slatecolor.266','slatecolor','slate color',[85,85,85],'555555',5592405],
['nbs-iscc-r:slategray.266','slategray','slate-gray',[85,85,85],'555555',5592405],
['nbs-iscc-r:slateolive.155','slateolive','slate-olive',[125,137,132],'7d8984',8227204],
['nbs-iscc-r:slatepurple.228','slatepurple','slate-purple',[121,104,120],'796878',7956600],
['nbs-iscc-r:slatepurple.245','slatepurple','slate-purple',[131,100,121],'836479',8610937],
['nbs-iscc-r:slateviolet.229','slateviolet','slate-violet ',[80,64,77],'50404d',5259341],
['nbs-iscc-r:slateviolet.234','slateviolet','slate-violet ',[93,85,91],'5d555b',6116699],
['nbs-iscc-r:slateviolet.228','slateviolet','slate-violet ',[121,104,120],'796878',7956600],
['nbs-iscc-r:smaltblue.194','smaltblue','smalt blue',[48,38,122],'30267a',3155578],
['nbs-iscc-r:smokegray.90','smokegray','smoke gray',[194,178,128],'c2b280',12759680],
['nbs-iscc-r:smokegray.93','smokegray','smoke gray',[191,184,165],'bfb8a5',12564645],
['nbs-iscc-r:snuffbrown.77','snuffbrown','snuff brown',[130,102,68],'826644',8545860],
['nbs-iscc-r:snuffbrown.80','snuffbrown','snuff brown',[126,109,90],'7e6d5a',8285530],
['nbs-iscc-r:softblueviolet.206','softblueviolet','soft blue-violet',[126,115,184],'7e73b8',8287160],
['nbs-iscc-r:softbluishviolet.207','softbluishviolet','soft bluish violet',[96,78,151],'604e97',6311575],
['nbs-iscc-r:sootyblack.266','sootyblack','sooty black',[85,85,85],'555555',5592405],
['nbs-iscc-r:sorghumbrown.46','sorghumbrown','sorghum brown',[103,76,71],'674c47',6769735],
['nbs-iscc-r:sorrentogreen.164','sorrentogreen','sorrento green',[49,120,115],'317873',3242099],
['nbs-iscc-r:spectrumblue.176','spectrumblue','spectrum blue',[0,161,194],'00a1c2',41410],
['nbs-iscc-r:spectrumred.11','spectrumred','spectrum red',[190,0,50],'be0032',12451890],
['nbs-iscc-r:spectrumviolet.207','spectrumviolet','spectrum violet',[96,78,151],'604e97',6311575],
['nbs-iscc-r:spinachgreen.120','spinachgreen','spinach green',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-r:spinelpink.255','spinelpink','spinel pink',[179,68,108],'b3446c',11748460],
['nbs-iscc-r:spinelred.255','spinelred','spinel red',[179,68,108],'b3446c',11748460],
['nbs-iscc-r:squillblue.181','squillblue','squill blue',[112,163,204],'70a3cc',7381964],
['nbs-iscc-r:stonegreen.136','stonegreen','stone green',[103,146,103],'679267',6787687],
['nbs-iscc-r:stormgray.155','stormgray','storm gray',[125,137,132],'7d8984',8227204],
['nbs-iscc-r:strawberrypink.37','strawberrypink','strawberry pink',[203,109,81],'cb6d51',13331793],
['nbs-iscc-r:strawyellow.86','strawyellow','straw yellow',[248,222,126],'f8de7e',16309886],
['nbs-iscc-r:strontianyellow.99','strontianyellow','strontian yellow ',[190,183,46],'beb72e',12498734],
['nbs-iscc-r:sudanbrown.55','sudanbrown','sudan brown',[128,70,27],'80461b',8406555],
['nbs-iscc-r:sudanbrown.57','sudanbrown','sudan brown',[166,123,91],'a67b5b',10910555],
['nbs-iscc-r:sudanbrown.74','sudanbrown','sudan brown',[153,101,21],'996515',10052885],
['nbs-iscc-r:sulphategreen.163','sulphategreen','sulphate green',[102,173,164],'66ada4',6729124],
['nbs-iscc-r:sulphategreen.164','sulphategreen','sulphate green',[49,120,115],'317873',3242099],
['nbs-iscc-r:sulphineyellow.88','sulphineyellow','sulphine yellow',[171,145,68],'ab9144',11243844],
['nbs-iscc-r:sulphuryellow.101','sulphuryellow','sulphur yellow',[234,230,121],'eae679',15394425],
['nbs-iscc-r:sulphuryellow.104','sulphuryellow','sulphur yellow',[235,232,164],'ebe8a4',15460516],
['nbs-iscc-r:taupebrown.229','taupebrown','taupe brown',[80,64,77],'50404d',5259341],
['nbs-iscc-r:tawny.54','tawny','tawny',[174,105,56],'ae6938',11430200],
['nbs-iscc-r:tawnyolive.77','tawnyolive','tawny-olive',[130,102,68],'826644',8545860],
['nbs-iscc-r:tawnyolive.94','tawnyolive','tawny-olive',[150,113,23],'967117',9859351],
['nbs-iscc-r:teagreen.122','teagreen','tea green',[143,151,121],'8f9779',9410425],
['nbs-iscc-r:terracotta.39','terracotta','terra cotta',[180,116,94],'b4745e',11826270],
['nbs-iscc-r:terreverte.150','terreverte','terre verte',[94,113,106],'5e716a',6189418],
['nbs-iscc-r:testaceous.39','testaceous','testaceous',[180,116,94],'b4745e',11826270],
['nbs-iscc-r:testaceous.42','testaceous','testaceous',[168,124,109],'a87c6d',11041901],
['nbs-iscc-r:thulitepink.247','thulitepink','thulite pink',[230,143,172],'e68fac',15110060],
['nbs-iscc-r:tibergreen.135','tibergreen','tiber green',[147,197,146],'93c592',9684370],
['nbs-iscc-r:tilleulbuff.31','tilleulbuff','tilleul-buff',[236,213,197],'ecd5c5',15521221],
['nbs-iscc-r:tourmalinepink.237','tourmalinepink','tourmaline pink',[158,79,136],'9e4f88',10375048],
['nbs-iscc-r:tourmalinepink.248','tourmalinepink','tourmaline pink',[222,111,161],'de6fa1',14577569],
['nbs-iscc-r:turquoisegreen.163','turquoisegreen','turquoise green',[102,173,164],'66ada4',6729124],
['nbs-iscc-r:turtlegreen.120','turtlegreen','turtle green',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-r:tyrianblue.186','tyrianblue','tyrian blue',[83,104,120],'536878',5466232],
['nbs-iscc-r:tyrianblue.204','tyrianblue','tyrian blue',[76,81,109],'4c516d',5001581],
['nbs-iscc-r:tyrianpink.237','tyrianpink','tyrian pink',[158,79,136],'9e4f88',10375048],
['nbs-iscc-r:tyrianrose.254','tyrianrose','tyrian rose',[206,70,118],'ce4676',13518454],
['nbs-iscc-r:tyrolitegreen.163','tyrolitegreen','tyrolite green',[102,173,164],'66ada4',6729124],
['nbs-iscc-r:ultramarineash.178','ultramarineash','ultramarine ash',[0,103,165],'0067a5',26533],
['nbs-iscc-r:ultramarineblue.176','ultramarineblue','ultramarine blue',[0,161,194],'00a1c2',41410],
['nbs-iscc-r:uraniablue.212','uraniablue','urania blue',[47,33,64],'2f2140',3088704],
['nbs-iscc-r:uraniablue.215','uraniablue','urania blue',[85,76,105],'554c69',5590121],
['nbs-iscc-r:uraniablue.229','uraniablue','urania blue',[80,64,77],'50404d',5259341],
['nbs-iscc-r:vanderpoelsblue.178','vanderpoelsblue','vanderpoel\'s blue',[0,103,165],'0067a5',26533],
['nbs-iscc-r:vanderpoelsblue.182','vanderpoelsblue','vanderpoel\'s blue',[67,107,149],'436b95',4418453],
['nbs-iscc-r:vanderpoelsgreen.116','vanderpoelsgreen','vanderpoel\'s green',[189,218,87],'bdda57',12442199],
['nbs-iscc-r:vanderpoelsgreen.117','vanderpoelsgreen','vanderpoel\'s green',[126,159,46],'7e9f2e',8298286],
['nbs-iscc-r:vanderpoelsviolet.210','vanderpoelsviolet','vanderpoel\'s violet',[140,130,182],'8c82b6',9208502],
['nbs-iscc-r:vandykebrown.58','vandykebrown','vandyke brown',[111,78,55],'6f4e37',7294519],
['nbs-iscc-r:vandykebrown.61','vandykebrown','vandyke brown',[99,81,71],'635147',6508871],
['nbs-iscc-r:vandykered.19','vandykered','vandyke red',[144,93,93],'905d5d',9461085],
['nbs-iscc-r:variscitegreen.143','variscitegreen','variscite green',[142,209,178],'8ed1b2',9359794],
['nbs-iscc-r:varleysgray.228','varleysgray','varley\'s gray',[121,104,120],'796878',7956600],
['nbs-iscc-r:varleysgreen.137','varleysgreen','varley\'s green',[53,94,59],'355e3b',3497531],
['nbs-iscc-r:venetianblue.178','venetianblue','venetian blue',[0,103,165],'0067a5',26533],
['nbs-iscc-r:venetianpink.26','venetianpink','venetian pink',[248,131,121],'f88379',16286585],
['nbs-iscc-r:venicegreen.159','venicegreen','venice green',[0,166,147],'00a693',42643],
['nbs-iscc-r:verbenaviolet.214','verbenaviolet','verbena violet',[150,144,171],'9690ab',9867435],
['nbs-iscc-r:verdigrisgreen.144','verdigrisgreen','verdigris green',[106,171,142],'6aab8e',6990734],
['nbs-iscc-r:vernoniapurple.262','vernoniapurple','vernonia purple',[145,95,109],'915f6d',9527149],
['nbs-iscc-r:veronabrown.58','veronabrown','verona brown',[111,78,55],'6f4e37',7294519],
['nbs-iscc-r:veronesegreen.119','veronesegreen','veronese green',[201,220,137],'c9dc89',13229193],
['nbs-iscc-r:veronesegreen.120','veronesegreen','veronese green',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-r:vetivergreen.120','vetivergreen','vetiver green',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-r:vetivergreen.122','vetivergreen','vetiver green',[143,151,121],'8f9779',9410425],
['nbs-iscc-r:victorialake.16','victorialake','victoria lake',[114,47,55],'722f37',7483191],
['nbs-iscc-r:victorialake.17','victorialake','victoria lake',[63,23,40],'3f1728',4134696],
['nbs-iscc-r:vinaceous.2','vinaceous','vinaceous',[234,147,153],'ea9399',15373209],
['nbs-iscc-r:vinaceous.3','vinaceous','vinaceous',[228,113,122],'e4717a',14971258],
['nbs-iscc-r:vinaceous.5','vinaceous','vinaceous',[222,165,164],'dea5a4',14591396],
['nbs-iscc-r:vinaceous.6','vinaceous','vinaceous',[192,128,129],'c08081',12615809],
['nbs-iscc-r:vinaceousbrown.19','vinaceousbrown','vinaceous-brown',[144,93,93],'905d5d',9461085],
['nbs-iscc-r:vinaceousbuff.28','vinaceousbuff','vinaceous-buff',[244,194,194],'f4c2c2',16040642],
['nbs-iscc-r:vinaceousbuff.29','vinaceousbuff','vinaceous-buff',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-r:vinaceousbuff.31','vinaceousbuff','vinaceous-buff',[236,213,197],'ecd5c5',15521221],
['nbs-iscc-r:vinaceousbuff.33','vinaceousbuff','vinaceous-buff',[194,172,153],'c2ac99',12758169],
['nbs-iscc-r:vinaceouscinnamon.29','vinaceouscinnamon','vinaceous-cinnamon',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-r:vinaceouscinnamon.76','vinaceouscinnamon','vinaceous-cinnamon',[193,154,107],'c19a6b',12687979],
['nbs-iscc-r:vinaceousdrab.19','vinaceousdrab','vinaceous-drab',[144,93,93],'905d5d',9461085],
['nbs-iscc-r:vinaceousfawn.29','vinaceousfawn','vinaceous-fawn',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-r:vinaceousfawn.32','vinaceousfawn','vinaceous-fawn',[199,173,163],'c7ada3',13086115],
['nbs-iscc-r:vinaceousgray.227','vinaceousgray','vinaceous-gray',[170,152,169],'aa98a9',11180201],
['nbs-iscc-r:vinaceousgray.228','vinaceousgray','vinaceous-gray',[121,104,120],'796878',7956600],
['nbs-iscc-r:vinaceouslavender.5','vinaceouslavender','vinaceous-lavender',[222,165,164],'dea5a4',14591396],
['nbs-iscc-r:vinaceouslilac.251','vinaceouslilac','vinaceous-lilac',[193,126,145],'c17e91',12680849],
['nbs-iscc-r:vinaceouslilac.262','vinaceouslilac','vinaceous-lilac',[145,95,109],'915f6d',9527149],
['nbs-iscc-r:vinaceouspink.39','vinaceouspink','vinaceous-pink',[180,116,94],'b4745e',11826270],
['nbs-iscc-r:vinaceouspink.42','vinaceouspink','vinaceous-pink',[168,124,109],'a87c6d',11041901],
['nbs-iscc-r:vinaceouspurple.258','vinaceouspurple','vinaceous-purple ',[168,81,110],'a8516e',11030894],
['nbs-iscc-r:vinaceouspurple.262','vinaceouspurple','vinaceous-purple ',[145,95,109],'915f6d',9527149],
['nbs-iscc-r:vinaceouspurple.245','vinaceouspurple','vinaceous-purple ',[131,100,121],'836479',8610937],
['nbs-iscc-r:vinaceousrufous.38','vinaceousrufous','vinaceous-rufous',[158,71,50],'9e4732',10372914],
['nbs-iscc-r:vinaceousrusset.42','vinaceousrusset','vinaceous-russet',[168,124,109],'a87c6d',11041901],
['nbs-iscc-r:vinaceousrusset.43','vinaceousrusset','vinaceous-russet',[121,68,59],'79443b',7947323],
['nbs-iscc-r:vinaceousslate.228','vinaceousslate','vinaceous-slate',[121,104,120],'796878',7956600],
['nbs-iscc-r:vinaceoustawny.39','vinaceoustawny','vinaceous-tawny',[180,116,94],'b4745e',11826270],
['nbs-iscc-r:violetcarmine.259','violetcarmine','violet carmine ',[103,49,71],'673147',6762823],
['nbs-iscc-r:violetgray.233','violetgray','violet-gray',[139,133,137],'8b8589',9143689],
['nbs-iscc-r:violetplumbeous.203','violetplumbeous','violet-plumbeous',[140,146,172],'8c92ac',9212588],
['nbs-iscc-r:violetpurple.216','violetpurple','violet-purple ',[154,78,174],'9a4eae',10112686],
['nbs-iscc-r:violetslate.186','violetslate','violet-slate',[83,104,120],'536878',5466232],
['nbs-iscc-r:violetultramarine.196','violetultramarine','violet ultramarine',[84,90,167],'545aa7',5528231],
['nbs-iscc-r:violetultramarine.207','violetultramarine','violet ultramarine',[96,78,151],'604e97',6311575],
['nbs-iscc-r:viridiangreen.140','viridiangreen','viridian green',[62,180,137],'3eb489',4109449],
['nbs-iscc-r:viridiangreen.141','viridiangreen','viridian green',[0,121,89],'007959',31065],
['nbs-iscc-r:viridinegreen.116','viridinegreen','viridine green',[189,218,87],'bdda57',12442199],
['nbs-iscc-r:viridineyellow.99','viridineyellow','viridine yellow',[190,183,46],'beb72e',12498734],
['nbs-iscc-r:viridineyellow.102','viridineyellow','viridine yellow',[185,180,89],'b9b459',12170329],
['nbs-iscc-r:vividgreen.140','vividgreen','vivid green',[62,180,137],'3eb489',4109449],
['nbs-iscc-r:wallgreen.164','wallgreen','wall green',[49,120,115],'317873',3242099],
['nbs-iscc-r:walnutbrown.43','walnutbrown','walnut brown',[121,68,59],'79443b',7947323],
['nbs-iscc-r:walnutbrown.46','walnutbrown','walnut brown',[103,76,71],'674c47',6769735],
['nbs-iscc-r:warblergreen.106','warblergreen','warbler green',[134,126,54],'867e36',8814134],
['nbs-iscc-r:warmblackishbrown.20','warmblackishbrown','warm blackish brown ',[84,61,63],'543d3f',5520703],
['nbs-iscc-r:warmblackishbrown.23','warmblackishbrown','warm blackish brown ',[92,80,79],'5c504f',6049871],
['nbs-iscc-r:warmbuff.71','warmbuff','warm buff ',[227,168,87],'e3a857',14919767],
['nbs-iscc-r:warmbuff.73','warmbuff','warm buff ',[250,214,165],'fad6a5',16438949],
['nbs-iscc-r:warmbuff.87','warmbuff','warm buff ',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-r:warmsepia.61','warmsepia','warm sepia ',[99,81,71],'635147',6508871],
['nbs-iscc-r:watergreen.121','watergreen','water green',[218,223,183],'dadfb7',14344119],
['nbs-iscc-r:waxyellow.84','waxyellow','wax yellow',[212,175,55],'d4af37',13938487],
['nbs-iscc-r:wedgewoodblue.180','wedgewoodblue','wedgewood blue',[161,202,241],'a1caf1',10603249],
['nbs-iscc-r:wedgewoodblue.181','wedgewoodblue','wedgewood blue',[112,163,204],'70a3cc',7381964],
['nbs-iscc-r:wedgewoodblue.198','wedgewoodblue','wedgewood blue',[179,188,226],'b3bce2',11779298],
['nbs-iscc-r:wedgewoodblue.199','wedgewoodblue','wedgewood blue',[135,145,191],'8791bf',8884671],
['nbs-iscc-r:white.263','white','white',[242,243,244],'f2f3f4',15922164],
['nbs-iscc-r:windsorblue.200','windsorblue','windsor blue',[78,81,128],'4e5180',5132672],
['nbs-iscc-r:wintergreen.136','wintergreen','winter green',[103,146,103],'679267',6787687],
['nbs-iscc-r:wistariablue.195','wistariablue','wistaria blue',[108,121,184],'6c79b8',7109048],
['nbs-iscc-r:wistariablue.206','wistariablue','wistaria blue',[126,115,184],'7e73b8',8287160],
['nbs-iscc-r:wistariaviolet.206','wistariaviolet','wistaria violet',[126,115,184],'7e73b8',8287160],
['nbs-iscc-r:woodbrown.76','woodbrown','wood brown',[193,154,107],'c19a6b',12687979],
['nbs-iscc-r:woodbrown.79','woodbrown','wood brown',[174,155,130],'ae9b82',11443074],
['nbs-iscc-r:xanthineorange.51','xanthineorange','xanthine orange',[190,101,22],'be6516',12477718],
['nbs-iscc-r:yaleblue.181','yaleblue','yale blue',[112,163,204],'70a3cc',7381964],
['nbs-iscc-r:yellowgreen.116','yellowgreen','yellow green',[189,218,87],'bdda57',12442199],
['nbs-iscc-r:yellowishcitrine.103','yellowishcitrine','yellowish citrine',[152,148,62],'98943e',9999422],
['nbs-iscc-r:yellowishglaucous.121','yellowishglaucous','yellowish glaucous',[218,223,183],'dadfb7',14344119],
['nbs-iscc-r:yellowishoilgreen.106','yellowishoilgreen','yellowish oil green',[134,126,54],'867e36',8814134],
['nbs-iscc-r:yellowisholive.106','yellowisholive','yellowish olive',[134,126,54],'867e36',8814134],
['nbs-iscc-r:yellowisholive.107','yellowisholive','yellowish olive',[102,93,30],'665d1e',6708510],
['nbs-iscc-r:yellowochre.72','yellowochre','yellow ochre',[190,138,61],'be8a3d',12487229],
['nbs-iscc-r:yewgreen.125','yewgreen','yew green',[74,93,35],'4a5d23',4873507],
['nbs-iscc-r:yvetteviolet.215','yvetteviolet','yvette violet',[85,76,105],'554c69',5590121],
['nbs-iscc-r:zincgreen.145','zincgreen','zinc green',[59,120,97],'3b7861',3897441],
['nbs-iscc-r:zincorange.50','zincorange','zinc orange',[237,135,45],'ed872d',15566637],
['nbs-iscc-r:zincorange.53','zincorange','zinc orange',[217,144,88],'d99058',14258264]
    ];
}

sub _description {
    return {
          'subtitle' => 'Color Standards and Color Nomenclature',
          'title' => 'NBS/ISCC R',
          'description' => '    Kenneth L. Kelly and Deanne B. Judd.
    "Color: Universal Language and Dictionary of Names",
    National Bureau of Standards,
    Spec. Publ. 440, Dec. 1976, 189 pages.


Color Name Dictionary: (R) Color Standards and Color Nomenclature

Ridgway,
Color Standards and Color Nomenclature 

While using some modifiers, this primarily idiosyncratic dictionary was intended for biology and botany. It maps 1096 names to 1602 colors, using 214 of the 267 centroids.

[http://swissnet.ai.mit.edu/~jaffer/Color/Dictionaries#R]
'
        }

}

1;
