"use strict";
// Copyright Amazon.com Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
Object.defineProperty(exports, "__esModule", { value: true });
exports.RawAesKeyringNode = void 0;
const material_management_node_1 = require("@aws-crypto/material-management-node");
const crypto_1 = require("crypto");
const serialize_1 = require("@aws-crypto/serialize");
const raw_keyring_1 = require("@aws-crypto/raw-keyring");
const fromUtf8 = (input) => Buffer.from(input, 'utf8');
const toUtf8 = (input) => Buffer.from(input.buffer, input.byteOffset, input.byteLength).toString('utf8');
const { serializeEncryptionContext } = (0, serialize_1.serializeFactory)(fromUtf8);
const { rawAesEncryptedDataKey } = (0, raw_keyring_1.rawAesEncryptedDataKeyFactory)(toUtf8, fromUtf8);
const { rawAesEncryptedParts } = (0, raw_keyring_1.rawAesEncryptedPartsFactory)(fromUtf8);
class RawAesKeyringNode extends material_management_node_1.KeyringNode {
    constructor(input) {
        super();
        const { keyName, keyNamespace, unencryptedMasterKey, wrappingSuite } = input;
        /* Precondition: AesKeyringNode needs identifying information for encrypt and decrypt. */
        (0, material_management_node_1.needs)(keyName && keyNamespace, 'Identifying information must be defined.');
        /* Precondition: RawAesKeyringNode requires wrappingSuite to be a valid RawAesWrappingSuite. */
        const wrappingMaterial = new raw_keyring_1.NodeRawAesMaterial(wrappingSuite)
            /* Precondition: unencryptedMasterKey must correspond to the NodeAlgorithmSuite specification.
             * Note: the KeyringTrace and flag are _only_ set because I am reusing an existing implementation.
             * See: raw_aes_material.ts in @aws-crypto/raw-keyring for details
             */
            .setUnencryptedDataKey(unencryptedMasterKey, {
            keyNamespace,
            keyName,
            flags: material_management_node_1.KeyringTraceFlag.WRAPPING_KEY_GENERATED_DATA_KEY,
        });
        const _wrapKey = async (material) => {
            /* The AAD section is uInt16BE(length) + AAD
             * see: https://docs.aws.amazon.com/encryption-sdk/latest/developer-guide/message-format.html#header-aad
             * However, the RAW Keyring wants _only_ the ADD.
             * So, I just slice off the length.
             */
            const { buffer, byteOffset, byteLength } = serializeEncryptionContext(material.encryptionContext).slice(2);
            const aad = Buffer.from(buffer, byteOffset, byteLength);
            const { keyNamespace, keyName } = this;
            return aesGcmWrapKey(keyNamespace, keyName, material, aad, wrappingMaterial);
        };
        const _unwrapKey = async (material, edk) => {
            const { keyNamespace, keyName } = this;
            /* The AAD section is uInt16BE(length) + AAD
             * see: https://docs.aws.amazon.com/encryption-sdk/latest/developer-guide/message-format.html#header-aad
             * However, the RAW Keyring wants _only_ the ADD.
             * So, I just slice off the length.
             */
            const { buffer, byteOffset, byteLength } = serializeEncryptionContext(material.encryptionContext).slice(2);
            const aad = Buffer.from(buffer, byteOffset, byteLength);
            // const aad = Buffer.concat(encodeEncryptionContext(context || {}))
            return aesGcmUnwrapKey(keyNamespace, keyName, material, wrappingMaterial, edk, aad);
        };
        (0, material_management_node_1.readOnlyProperty)(this, 'keyName', keyName);
        (0, material_management_node_1.readOnlyProperty)(this, 'keyNamespace', keyNamespace);
        (0, material_management_node_1.readOnlyProperty)(this, '_wrapKey', _wrapKey);
        (0, material_management_node_1.readOnlyProperty)(this, '_unwrapKey', _unwrapKey);
    }
    _filter({ providerId, providerInfo }) {
        const { keyNamespace, keyName } = this;
        return providerId === keyNamespace && providerInfo.startsWith(keyName);
    }
    _onEncrypt = (0, raw_keyring_1._onEncrypt)(randomBytesAsync);
    _onDecrypt = (0, raw_keyring_1._onDecrypt)();
}
exports.RawAesKeyringNode = RawAesKeyringNode;
(0, material_management_node_1.immutableClass)(RawAesKeyringNode);
const encryptFlags = material_management_node_1.KeyringTraceFlag.WRAPPING_KEY_ENCRYPTED_DATA_KEY |
    material_management_node_1.KeyringTraceFlag.WRAPPING_KEY_SIGNED_ENC_CTX;
const decryptFlags = material_management_node_1.KeyringTraceFlag.WRAPPING_KEY_DECRYPTED_DATA_KEY |
    material_management_node_1.KeyringTraceFlag.WRAPPING_KEY_VERIFIED_ENC_CTX;
/**
 * Uses aes-gcm to encrypt the data key and return the passed NodeEncryptionMaterial with
 * an EncryptedDataKey added.
 * @param keyNamespace [String] The keyring namespace (for KeyringTrace)
 * @param keyName [String] The keyring name (for KeyringTrace and to extract the extra info stored in providerInfo)
 * @param material [NodeEncryptionMaterial] The target material to which the EncryptedDataKey will be added
 * @param aad [Buffer] The serialized aad (EncryptionContext)
 * @param wrappingMaterial [NodeRawAesMaterial] The material used to decrypt the EncryptedDataKey
 * @returns [NodeEncryptionMaterial] Mutates and returns the same NodeEncryptionMaterial that was passed but with an EncryptedDataKey added
 */
function aesGcmWrapKey(keyNamespace, keyName, material, aad, wrappingMaterial) {
    const { encryption, ivLength } = wrappingMaterial.suite;
    const iv = (0, crypto_1.randomBytes)(ivLength);
    const wrappingDataKey = wrappingMaterial.getUnencryptedDataKey();
    const dataKey = (0, material_management_node_1.unwrapDataKey)(material.getUnencryptedDataKey());
    const cipher = (0, crypto_1.createCipheriv)(encryption, wrappingDataKey, iv).setAAD(aad);
    // Buffer.concat will use the shared buffer space, and the resultant buffer will have a byteOffset...
    const ciphertext = (0, serialize_1.concatBuffers)(cipher.update(dataKey), cipher.final());
    const authTag = cipher.getAuthTag();
    const edk = rawAesEncryptedDataKey(keyNamespace, keyName, iv, ciphertext, authTag);
    return material.addEncryptedDataKey(edk, encryptFlags);
}
/**
 * Uses aes-gcm to decrypt the encrypted data key and return the passed NodeDecryptionMaterial with
 * the unencrypted data key set.
 * @param keyNamespace [String] The keyring namespace (for KeyringTrace)
 * @param keyName [String] The keyring name (for KeyringTrace and to extract the extra info stored in providerInfo)
 * @param material [NodeDecryptionMaterial] The target material to which the decrypted data key will be added
 * @param wrappingMaterial [NodeRawAesMaterial] The material used to decrypt the EncryptedDataKey
 * @param edk [EncryptedDataKey] The EncryptedDataKey on which to operate
 * @param aad [Buffer] The serialized aad (EncryptionContext)
 * @returns [NodeDecryptionMaterial] Mutates and returns the same NodeDecryptionMaterial that was passed but with the unencrypted data key set
 */
function aesGcmUnwrapKey(keyNamespace, keyName, material, wrappingMaterial, edk, aad) {
    const { authTag, ciphertext, iv } = rawAesEncryptedParts(material.suite, keyName, edk);
    const { encryption } = wrappingMaterial.suite;
    // createDecipheriv is incorrectly typed in @types/node. It should take key: CipherKey, not key: BinaryLike
    const decipher = (0, crypto_1.createDecipheriv)(encryption, wrappingMaterial.getUnencryptedDataKey(), iv)
        .setAAD(aad)
        .setAuthTag(authTag);
    // Buffer.concat will use the shared buffer space, and the resultant buffer will have a byteOffset...
    const unencryptedDataKey = (0, serialize_1.concatBuffers)(decipher.update(ciphertext), decipher.final());
    const trace = { keyNamespace, keyName, flags: decryptFlags };
    return material.setUnencryptedDataKey(unencryptedDataKey, trace);
}
async function randomBytesAsync(size) {
    return new Promise((resolve, reject) => {
        (0, crypto_1.randomBytes)(size, (err, buffer) => {
            if (err)
                return reject(err);
            resolve(buffer);
        });
    });
}
//# sourceMappingURL=data:application/json;base64,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